<?php

namespace Drupal\entity_bundle_scaffold\Hooks;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_bundle_scaffold\Service\Generator\ControllerClassGenerator;
use Drupal\entity_bundle_scaffold\Service\Generator\EntityBundleClassGenerator;
use Drupal\field\Entity\FieldConfig;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Hook implementations for entity insert events.
 */
class EntityInsertHooks {

  use StringTranslationTrait;

  /**
   * The filesystem.
   */
  protected Filesystem $fileSystem;

  /**
   * EntityInsertHooks constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\entity_bundle_scaffold\Service\Generator\EntityBundleClassGenerator $bundleClassGenerator
   *   The entity bundle class generator.
   * @param \Drupal\entity_bundle_scaffold\Service\Generator\ControllerClassGenerator $controllerClassGenerator
   *   The controller class generator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
    protected MessengerInterface $messenger,
    protected EntityBundleClassGenerator $bundleClassGenerator,
    protected ControllerClassGenerator $controllerClassGenerator
  ) {
    $this->fileSystem = new Filesystem();
  }

  /**
   * Runs when a field config entity is inserted.
   */
  public function onFieldConfigInsert(FieldConfig $entity): void {
    $this->appendFieldGetter($entity);
  }

  /**
   * Runs when a bundle config entity is inserted.
   */
  public function onBundleInsert(ConfigEntityInterface $entity): void {
    $this->generateBundleClass($entity);
    $this->generateController($entity);
  }

  /**
   * Append a getter to the bundle class.
   *
   * @param \Drupal\field\Entity\FieldConfig $entity
   *   The field config entity.
   */
  protected function appendFieldGetter(FieldConfig $entity): void {
    $entityTypeId = $entity->getTargetEntityTypeId();
    $bundle = $entity->getTargetBundle();
    $settings = $this->configFactory->get('entity_bundle_scaffold.settings');

    $autoUpdate = $settings->get('generators.bundle_class.auto_update');
    if (!$autoUpdate) {
      return;
    }

    $module = $settings->get('generators.bundle_class.output_module');
    if ($module === NULL) {
      return;
    }

    if (!$output = $this->bundleClassGenerator->appendFieldGettersToExistingClass($entityTypeId, $bundle, [$entity])) {
      return;
    }

    $destination = $this->bundleClassGenerator->getEntityBundleClassPath($entityTypeId, $bundle)
      ?? $this->bundleClassGenerator->buildEntityBundleClassPath($entityTypeId, $bundle, $module);

    if (!file_exists($destination)) {
      return;
    }

    $this->fileSystem->remove($destination);
    $this->fileSystem->appendToFile($destination, $output);

    $bundleClassName = $this->entityTypeManager
      ->getStorage($entityTypeId)
      ->getEntityClass($bundle);

    $this->messenger->addStatus($this->t('Added new getter for @fieldName to @bundleClassName', [
      '@fieldName' => $entity->getName(),
      '@bundleClassName' => $bundleClassName,
    ]));
  }

  /**
   * Generate a bundle class.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The bundle entity.
   */
  protected function generateBundleClass(ConfigEntityInterface $entity): void {
    $entityType = $entity->getEntityType()->getBundleOf();
    if ($entityType === NULL) {
      return;
    }

    $bundle = $entity->id();
    $settings = $this->configFactory->get('entity_bundle_scaffold.settings');

    $autoCreate = $settings->get('generators.bundle_class.auto_create');
    if (!$autoCreate) {
      return;
    }

    $module = $settings->get('generators.bundle_class.output_module');
    if ($module === NULL) {
      return;
    }

    $definition = $this->entityTypeManager
      ->getDefinition($entityType);
    $existingClassName = $this->entityTypeManager
      ->getStorage($entityType)
      ->getEntityClass($bundle);
    $hasExisting = FALSE;

    if ($existingClassName && $existingClassName !== $definition->getClass()) {
      $hasExisting = TRUE;
      $destination = (new \ReflectionClass($existingClassName))->getFileName();

      if (file_exists($destination)) {
        return;
      }

      $output = $this->bundleClassGenerator->generateExisting($entityType, $bundle);
    }
    else {
      $destination = $this->bundleClassGenerator->buildEntityBundleClassPath($entityType, $bundle, $module);
      $output = $this->bundleClassGenerator->generateNew($entityType, $bundle, $module);
    }

    $this->fileSystem->remove($destination);
    $this->fileSystem->appendToFile($destination, $output);

    $this->messenger->addStatus($this->t('Successfully @operation entity bundle class.', [
      '@operation' => $hasExisting ? $this->t('updated') : $this->t('created'),
    ]));

    if (!$this->bundleClassGenerator->areModelsAnnotationBased()) {
      $this->messenger->addStatus($this->t('Don\'t forget to register your bundle class through a <a href="@hookEntityBundleInfoAlterUrl">hook_entity_bundle_info_alter implementation</a>, or by using the <a href="@entityModelUrl">Entity Model module<a/>.', [
        '@hookEntityBundleInfoAlterUrl' => 'https://www.drupal.org/node/3191609',
        '@entityModelUrl' => 'https://www.drupal.org/project/entity_model',
      ]));
    }

    // Clear cached bundle classes.
    $this->bundleClassGenerator->clearEntityBundleCaches();
  }

  /**
   * Generate a controller class.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The bundle entity.
   */
  protected function generateController(ConfigEntityInterface $entity): void {
    $entityType = $entity->getEntityType()->getBundleOf();
    if ($entityType === NULL) {
      return;
    }

    $bundle = $entity->id();
    $settings = $this->configFactory->get('entity_bundle_scaffold.settings');

    $autoCreate = $settings->get('generators.controller.auto_create');
    if (!$autoCreate) {
      return;
    }

    $module = $settings->get('generators.controller.output_module');
    if ($module === NULL) {
      return;
    }

    $className = $this->controllerClassGenerator->buildClassName($entityType, $bundle, $module);
    $destination = $this->controllerClassGenerator->buildControllerPath($entityType, $bundle, $module);

    try {
      new \ReflectionClass($className);

      if (file_exists($destination)) {
        return;
      }
    }
    catch (\ReflectionException) {
      // Noop.
    }

    $output = $this->controllerClassGenerator->generateNew($entityType, $bundle, $module);
    $this->fileSystem->remove($destination);
    $this->fileSystem->appendToFile($destination, $output);

    $this->messenger->addStatus($this->t('Successfully created controller class.'));
  }

}
