<?php

namespace Drupal\entity_bundle_scaffold\Plugin\EntityBundleClassMethodGenerator;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\entity_bundle_scaffold\EntityBundleClassMethodGeneratorBase;
use PhpParser\Builder\Method;
use PhpParser\Node\Identifier;
use PhpParser\Node\NullableType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for enum item field method generators.
 */
abstract class BaseEnumItem extends EntityBundleClassMethodGeneratorBase {

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The field type plugin manager.
   *
   * @var \Drupal\Core\Field\FieldTypePluginManagerInterface
   */
  protected $fieldTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->fieldTypeManager = $container->get('plugin.manager.field.field_type');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildGetter(FieldDefinitionInterface $field, Method $method, array &$uses): void {
    $fieldEnumClass = $this->getEnumClass($field);
    $fieldEnumClass = new \ReflectionClass($fieldEnumClass);

    $uses[] = $this->builderFactory->use($fieldEnumClass->getName());

    $expression = $this->helper->isFieldMultiple($field)
      ? sprintf('return $this->get(\'%s\')->enums();', $field->getName())
      : sprintf('return $this->get(\'%s\')->enum;', $field->getName());

    if ($this->helper->isFieldMultiple($field)) {
      $method->setReturnType('array');
      $method->setDocComment(sprintf('/** @return %s[] */', $fieldEnumClass->getShortName()));
    }
    else {
      $method->setReturnType(new NullableType(new Identifier($fieldEnumClass->getShortName())));
    }

    $method->addStmts($this->helper->parseExpression($expression));
  }

  /**
   * Gets the enum class for the field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field
   *   The field definition.
   *
   * @return string
   *   The enum class.
   */
  abstract protected function getEnumClass(FieldDefinitionInterface $field): string;

}
