<?php

namespace Drupal\entity_change\Plugin;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Plugin\ContextAwarePluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Base class for Entity change plugins.
 */
abstract class EntityChangeBase extends ContextAwarePluginBase implements EntityChangeInterface {

  use StringTranslationTrait;

  //---------------------------------------------------------------------------

  /**
   * We use this to store the value, so we only have to test once.
   *
   * @var bool|null
   */
  protected ?bool $hasChanged = NULL;

  /**
   * Is this EntityChange plugin appropriate to the supplied context?
   *
   * @return bool
   */
  abstract public function applies(): bool;

  /**
   * @inheritDoc
   */
  public function changed(): bool {
    if ($this->hasChanged === NULL) {
      try {
        $new = $this->getContextValue('entity');
        $old = $this->getContextValue('original');
        $this->hasChanged = $this->hasChanged($new, $old);
      }
      catch (ContextException $e) {
        \Drupal::logger('entity_change')->error($this->t('Could not get context values: %x', ['%x' => $e->getMessage()]));
        $this->hasChanged = FALSE;
      }
    }
    return $this->hasChanged;
  }

  /**
   * Perform the plugin-specific check.
   *
   * @param EntityInterface $new
   * @param EntityInterface $old
   *
   * @return bool
   */
  abstract protected function hasChanged(EntityInterface $new, EntityInterface $old): bool;

}
