<?php

namespace Drupal\entity_change\Plugin;

use Drupal\Component\Plugin\CategorizingPluginManagerInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\CategorizingPluginManagerTrait;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_change\Plugin\Attribute\EntityChange;

/**
 * Provides the Entity change plugin manager.
 */
class EntityChangeManager extends DefaultPluginManager implements CategorizingPluginManagerInterface {

  use CategorizingPluginManagerTrait;
  use StringTranslationTrait;
  use LoggerChannelTrait;

  /**
   * Constructs a new EntityChangeManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct('Plugin/EntityChange', $namespaces, $module_handler,
        plugin_interface:  EntityChangeInterface::class,
        plugin_definition_attribute_name: EntityChange::class,
        plugin_definition_annotation_name: \Drupal\entity_change\Annotation\EntityChange::class
    );

    $this->alterInfo('entity_change_entity_change_plugin_info');
    $this->setCacheBackend($cache_backend, 'entity_change_entity_change_plugin_plugins');
  }

  /**
   * Given the supplied entity, find all matching EntityChange plugins
   * and instantiate each one for this $entity and return them
   *
   * @param EntityInterface $entity
   *
   * @param EntityInterface $original
   *
   * @return EntityChangeBase[]
   */
  public function buildInstances(EntityInterface $entity, EntityInterface $original): array {
    /** @var EntityChangeBase[] $entityChanges */
    $entityChanges = [];

    $entityTypeId = $entity->getEntityTypeId();
    $entityBundle = $entity->bundle();
    foreach ($this->getDefinitions() as $pluginName => $definition) {
      if ($this->isMatchingDefinition($definition['type'], $entityTypeId, $entityBundle)) {
        // The basic type matches, so include this one.
        if ($entityChange = $this->getApplicableInstance($pluginName, $entity, $original)) {
          // If it applies, keep it.
          $entityChanges[$pluginName] = $entityChange;
        }
      }
    }

    return $entityChanges;
  }

  /**
   * Does the entity's type and bundle match the given type.
   *
   * Example types: 'node', '
   *
   * @param string $type
   * @param string $entityTypeId
   * @param string $bundle
   *
   * @return bool
   */
  protected function isMatchingDefinition(string $type, string $entityTypeId, string $bundle): bool {
    [$entityType, $entityBundle, ] = explode(':', $type . ':*');

    if ($entityType === '*') {
      // Match everything.
      return true;
    }

    if ($entityBundle === '*') {
      // Match all bundles of a type.
      return $entityType === $entityTypeId;
    }

    // Entity tpe and bundle must both match.
    return $entityType === $entityTypeId && $entityBundle === $bundle;
  }

  /**
   * Get a specific instance of an entity change plugin, if it applies.
   *
   * @param string $pluginName
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\Core\Entity\EntityInterface $original
   *
   * @return \Drupal\entity_change\Plugin\EntityChangeBase|null
   */
  public function getApplicableInstance(string $pluginName, EntityInterface $entity, EntityInterface $original): ?EntityChangeBase {
    $logger = $this->getLogger('entity_change:manager');

    try {
      /** @var EntityChangeBase $entityChange */
      $entityChange = $this->createInstance($pluginName, ['context' => ['entity' => $entity, 'original' => $original]]);
      return $entityChange->applies() ? $entityChange : NULL;
    }
    catch (PluginException $e) {
      $logger->error($this->t('Unable to instantiate EntityChange @k: @x.', ['@k' => $pluginName, '@x' => $e->getMessage()]));
    }

    return NULL;
  }

  /**
   * Returns string of entity change definitions that match the entity by type.
   *
   * It does not instantiate any of the entity change plugins.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *
   * @return array
   */
  public function getMatchedDefinitions(EntityInterface $entity): array {
    /** @var string[] $matchedDefinitions */
    $matchedDefinitions = [];

    $entityTypeId = $entity->getEntityTypeId();
    $entityBundle = $entity->bundle();
    foreach ($this->getDefinitions() as $pluginName => $definition) {
      if ($this->isMatchingDefinition($definition['type'], $entityTypeId, $entityBundle)) {
        // The basic type matches, so include this one.
        $matchedDefinitions[$pluginName] = $definition;
      }
    }

    return $matchedDefinitions;
  }

}
