<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_contact_email\Functional;

use Drupal\entity_contact\Entity\EntityContactForm;
use Drupal\entity_contact_email\Entity\EntityContactEmail;
use Drupal\Tests\entity_contact\Functional\EntityContactTestBase;
use Drupal\Tests\field_ui\Traits\FieldUiTestTrait;

/**
 * Test the entity_contact_email module.
 *
 * @group entity_contact_email
 */
class EntityContactEmailTest extends EntityContactTestBase {

  use FieldUiTestTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'text',
    'entity_contact',
    'entity_contact_test',
    'entity_contact_email',
    'entity_contact_route',
    'field_ui',
    'block',
  ];

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->state = $this->container->get('state');
  }

  /**
   * Test the contact form route.
   */
  public function testSendEmail(): void {
    $this->drupalLogin($this->adminUser);

    // Create a form.
    $id = 'testform';
    $message = 'The form has been submitted.';
    $submit_button_text = 'Submit button text';

    $form = EntityContactForm::create([
      'id' => $id,
      'label' => $id,
      'message' => $message,
      'description' => [
        'value' => 'description',
      ],
      'submit_button_text' => $submit_button_text,
      'store_submissions' => FALSE,
      'remove_stored_submissions_after' => '',
    ]);
    $form->save();

    // Submit a form.
    $this->drupalGet('contact-form/' . $id);
    $this->submitForm([], $submit_button_text);

    $captured_emails = $this->state->get('system.test_mail_collector', []);
    $this->assertEmpty($captured_emails);

    // Create a e-mail.
    $email = EntityContactEmail::create([
      'entityContactForm' => $form->id(),
      'id' => 'test1',
      'label' => 'Test1',
      'subject' => 'Test 1 email subject',
      'body' => [
        'value' => 'Test 1 email body',
      ],
      'recipients' => [
        'recipient1@recipient1.com',
        'recipient2@recipient2.com',
      ],
    ]);
    $email->save();

    // Submit a form.
    $this->drupalGet('contact-form/' . $id);
    $this->submitForm([], $submit_button_text);

    // Two emails should be sent.
    $captured_emails = $this->state->get('system.test_mail_collector', []);
    $this->assertCount(2, $captured_emails);

    $this->assertEquals('recipient1@recipient1.com', $captured_emails[0]['to']);
    $this->assertStringContainsString('Test 1 email subject', $captured_emails[0]['subject']);
    $this->assertStringContainsString('Test 1 email body', $captured_emails[0]['body']);

    $this->assertEquals('recipient2@recipient2.com', $captured_emails[1]['to']);
    $this->assertStringContainsString('Test 1 email subject', $captured_emails[1]['subject']);
    $this->assertStringContainsString('Test 1 email body', $captured_emails[1]['body']);
  }

}
