<?php

declare(strict_types=1);

namespace Drupal\entity_extra_field_ui\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Define translate local task derivative.
 */
class TranslateLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Translate local task constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    $base_plugin_id,
  ): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition): array {
    $tasks = parent::getDerivativeDefinitions($base_plugin_definition);

    // Only add translate tasks if config_translation module is enabled.
    if (!$this->moduleHandler->moduleExists('config_translation')) {
      return $tasks;
    }

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $definition) {
      if ($entity_type_id !== 'node') {
        continue;
      }
      // Check if the entity type has a field UI base route set. If not, it is
      // not fieldable via UI:
      $base_route_name = $definition->get('field_ui_base_route');

      if (!isset($base_route_name)) {
        continue;
      }

      $translate_route = "entity.$entity_type_id.extra_fields.translate";
      $edit_route = "entity.$entity_type_id.extra_fields.edit";

      $tasks[$translate_route] = [
        'title' => $this->t('Translate'),
        'route_name' => $translate_route,
        'base_route' => $edit_route,
      ] + $base_plugin_definition;

      // Add a tab for the edit route itself, otherwise the "Translate" tab
      // won't be shown:
      $tasks[$edit_route] = [
        'title' => $this->t('Edit'),
        'route_name' => $edit_route,
        'base_route' => $edit_route,
      ] + $base_plugin_definition;
    }

    return $tasks;
  }

}
