<?php

declare(strict_types=1);

namespace Drupal\entity_extra_field\Plugin\ExtraFieldType;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Display\EntityDisplayInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\entity_extra_field\ExtraFieldTypePluginBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Utility\Token;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;

/**
 * Define extra field twig plugin.
 *
 * @ExtraFieldType(
 *   id = "twig",
 *   label = @Translation("Twig")
 * )
 */
class ExtraFieldTwigPlugin extends ExtraFieldTypePluginBase {
  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The theme manager service.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected ThemeManagerInterface $themeManager;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected PathMatcherInterface $pathMatcher;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * Constructs a ExtraFieldTwigPlugin object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $current_route_match
   *   The current route match service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager service.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   The path matcher service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    Token $token,
    ModuleHandlerInterface $module_handler,
    RouteMatchInterface $current_route_match,
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    RendererInterface $renderer,
    ThemeManagerInterface $theme_manager,
    PathMatcherInterface $path_matcher,
    LanguageManagerInterface $language_manager,
    AccountInterface $current_user,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $token,
      $module_handler,
      $current_route_match,
      $entity_type_manager,
      $entity_field_manager
    );
    $this->renderer = $renderer;
    $this->themeManager = $theme_manager;
    $this->pathMatcher = $path_matcher;
    $this->languageManager = $language_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('token'),
      $container->get('module_handler'),
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('renderer'),
      $container->get('theme.manager'),
      $container->get('path.matcher'),
      $container->get('language_manager'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'twig_template' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['twig_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Twig Template'),
      '#required' => TRUE,
      '#description' => $this->t('Provide the Twig code to render.<br> These are the context variables you can use in the template: @context.<br>The "entity" variable is special in that it represents the entity the extra field resides on. If you want to access the entity title, for example, you can use <code>{{ entity.title.value }}</code>.<br>Furthermore, you can alter the twig context through "hook_entity_extra_field_twig_context_alter." For an example implementation, check out "entity_extra_field.api.php".', ['@context' => implode(', ', array_keys($this->getContext()))]),
      '#default_value' => $this->getConfiguration()['twig_template'],
      '#rows' => 20,
    ];

    return $form;
  }

  /**
   * Validation callback for a Template element.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) : void {
    $build = [
      '#type' => 'inline_template',
      '#template' => $form_state->getValue('twig_template'),
      // We are not passing all the context, but that is fine, as using
      // unavailable context variables won't result in an exception:
      '#context' => $this->getContext(),
    ];
    try {
      $this->renderer->renderInIsolation($build);
    }
    catch (\Exception $exception) {
      $form_state->setError($form, $this->t(
        'Template error: @error',
        ['@error' => $exception->getMessage()])
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function build(EntityInterface $entity, EntityDisplayInterface $display): array {
    $build = [];

    if ($entity instanceof ContentEntityInterface) {
      // Get te twig template output:
      $twigTemplateOutput = $this->getConfiguration()['twig_template'];

      // Get the global twig context:
      $context = $this->getContext($entity);

      // Invoke module alter hook to modify the context:
      $this->moduleHandler->alter('entity_extra_field_twig_context', $context, $entity, $display);

      $build = [
        '#type' => 'inline_template',
        '#template' => $twigTemplateOutput,
        '#context' => $context,
      ];

    }
    return $build;
  }

  /**
   * Provides context variables for the twig template.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface|null $entity
   *   The entity object.
   *
   * @return array
   *   The context variables.
   */
  protected function getContext(?ContentEntityInterface $entity = NULL): array {
    $context = [];

    // Add the global context variables:
    $theme = $this->themeManager->getActiveTheme();
    $context['theme'] = $theme->getName();
    $context['theme_directory'] = $theme->getPath();

    $context['base_path'] = base_path();
    $context['front_page'] = Url::fromRoute('<front>');
    $context['is_front'] = $this->pathMatcher->isFrontPage();
    $context['language'] = $this->languageManager->getCurrentLanguage();

    $user = $this->currentUser;
    $context['is_admin'] = $user->hasPermission('access administration pages');
    $context['logged_in'] = $user->isAuthenticated();

    // Add the entity context, even if it is NULL, so it is displayed in the
    // twig_template description:
    $context['entity'] = $entity;

    return $context;
  }

}
