<?php

namespace Drupal\entity_io\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\entity_io\Service\EntityIoExport;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for exporting node revisions as JSON.
 */
class NodeRevisionExportController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity_io.export service.
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a NodeRevisionExportController object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityIoExport $entity_io_export, $exporter, LanguageManagerInterface $language_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityIoExport = $entity_io_export;
    $this->exporter = $exporter;
    $this->languageManager = $language_manager;
  }

  /**
   * Creates an instance of the controller.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_io.export'),
      $container->get('entity_io.exporter'),
      $container->get('language_manager')
    );
  }

  /**
   * Exports a specific node revision as JSON.
   */
  public function export(NodeInterface $node, $node_revision) {
    $revision = $this->entityTypeManager
      ->getStorage('node')
      ->loadRevision($node_revision);

    if (!$revision) {
      return new Response('Revision not found.', 404);
    }

    // Use injected entity_io.export service to get selected fields.
    $selected_fields = $this->entityIoExport->getSelectedFields($revision);

    // Use injected language manager.
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $max_depth = -1;
    $depth = 0;

    // Use injected exporter service; call static-like method on its class.
    $exporter = $this->exporter;
    $exporter::toJson($revision, $depth, $max_depth, $selected_fields, $current_langcode, TRUE);

    // Retrieve generated JSON and filename from exporter class properties.
    $json = $exporter::$json;
    $fileName = $exporter::$fileName;

    return new Response($json, 200, [
      'Content-Type' => 'application/json',
      'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
    ]);
  }

}
