<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\entity_io\Service\EntityIoExport;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to select and batch export entities.
 */
class EntityBatchExportForm extends FormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity_io.export service (concrete/instance).
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The exporter service.
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * Static references for use inside static batch callbacks.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|null
   */
  protected static $staticEntityTypeManager;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface|null
   */
  protected static $staticLanguageManager;


  /**
   * The entity_io.export service (concrete/instance).
   *
   * @var \Drupal\entity_io\Service\EntityIoExport|null
   */
  protected static $staticEntityIoExport;

  /**
   * The exporter service.
   *
   * @var mixed
   */
  protected static $staticExporter;

  /**
   * Constructs the form.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, EntityIoExport $entity_io_export, $exporter) {
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->entityIoExport = $entity_io_export;
    $this->exporter = $exporter;

    // Set static refs for static methods.
    self::$staticEntityTypeManager = $entity_type_manager;
    self::$staticLanguageManager = $language_manager;
    self::$staticEntityIoExport = $entity_io_export;
    self::$staticExporter = $exporter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('entity_io.export'),
      $container->get('entity_io.exporter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_entity_batch_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => [
        '' => $this->t('- Select -'),
        'node' => $this->t('Node'),
        'taxonomy_term' => $this->t('Taxonomy Term'),
        'block_content' => $this->t('Block'),
        'media' => $this->t('Media'),
        'user' => $this->t('User'),
      ],
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateBundleSelect',
        'wrapper' => 'bundle-wrapper',
      ],
      '#default_value' => $entity_type,
    ];

    $form['bundle_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'bundle-wrapper'],
    ];

    if (!empty($entity_type)) {
      $form['bundle_wrapper']['bundle'] = [
        '#type' => 'select',
        '#title' => $this->t('Select bundle'),
        '#options' => $this->getBundles($entity_type),
      ];
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export All'),
    ];

    return $form;
  }

  /**
   * Ajax callback to update the bundle select options.
   */
  public function updateBundleSelect(array &$form, FormStateInterface $form_state) {
    return $form['bundle_wrapper'];
  }

  /**
   * Get bundles for a given entity type.
   */
  protected function getBundles($entity_type) {
    $options = [];

    switch ($entity_type) {
      case 'node':
        // Use injected entity type manager to load node types.
        foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'taxonomy_term':
        foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocab) {
          $options[$vocab->id()] = $vocab->label();
        }
        break;

      case 'media':
        foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'block_content':
        foreach ($this->entityTypeManager->getStorage('block_content_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'user':
        $options = ['user' => 'User'];
        break;
    }

    return $options;
  }

  /**
   * Submit the export form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');
    $bundle = $form_state->getValue('bundle');

    // Use injected entity type manager instead of \Drupal::entityTypeManager().
    $query = $this->entityTypeManager->getStorage($entity_type)->getQuery();

    switch ($entity_type) {
      case 'node':
      case 'block_content':
        $query->condition('type', $bundle);
        break;

      case 'taxonomy_term':
        $query->condition('vid', $bundle);
        break;

      case 'media':
        $query->condition('bundle', $bundle);
        break;
    }

    $ids = $query->accessCheck(FALSE)->execute();

    $batch_builder = new BatchBuilder();
    $batch_builder->setTitle($this->t('Exporting entities'))
      ->setFinishCallback([static::class, 'batchFinished']);

    foreach ($ids as $id) {
      // Keep static callback for batch operations.
      $batch_builder->addOperation([static::class, 'exportEntityOperation'], [$entity_type, $id]);
    }

    batch_set($batch_builder->toArray());
  }

  /**
   * Export a single entity.
   *
   * Uses statically available services set in create() / constructor.
   */
  public static function exportEntityOperation($entity_type, $entity_id, array &$context) {
    // Use static injected entity type manager reference.
    $entity = NULL;
    if (self::$staticEntityTypeManager) {
      $entity = self::$staticEntityTypeManager->getStorage($entity_type)->load($entity_id);
    }
    else {
      // Fallback to service container if static ref isn't available.
      $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    }

    if ($entity) {
      // Obtain selected fields via injected/static entity_io.export.
      if (self::$staticEntityIoExport) {
        $selected_fields = self::$staticEntityIoExport::getSelectedFields($entity);
      }
      else {
        $selected_fields = \Drupal::service('entity_io.export')::getSelectedFields($entity);
      }

      $max_depth = -1;
      $depth = 0;

      // Use language manager (static if available).
      $current_langcode = self::$staticLanguageManager
        ? self::$staticLanguageManager->getCurrentLanguage()->getId()
        : \Drupal::languageManager()->getCurrentLanguage()->getId();

      // Use exporter service (static if available).
      $exporter = self::$staticExporter ?? \Drupal::service('entity_io.exporter');
      $exporter::toJson($entity, $depth, $max_depth, $selected_fields, $current_langcode);

      // Try to pick file URL from exporter.
      $file_url = $exporter::$publicUrl ?? ($exporter::$publicUrl ?? '');

      $context['results'][] = Markup::create('<a href="' . $file_url . '" target="_blank">' . $file_url . '</a>');
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function batchFinished($success, array $results, array $operations) {
    if ($success) {
      \Drupal::messenger()->addMessage(t('Export finished. Download files:'));
      foreach ($results as $url) {
        \Drupal::messenger()->addMessage($url);
      }
    }
    else {
      \Drupal::messenger()->addError(t('Export failed.'));
    }
  }

}
