<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to configure export path and behavior for JSON files.
 */
class EntityImportSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs the settings form.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_import_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['entity_io.import_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('entity_io.import_settings');

    // === Import settings fieldset ===
    $form['import_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Import settings'),
      '#description' => $this->t('Configure how nodes are handled during import operations.'),
    ];

    $form['import_settings']['import_create_new_revision'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create new revision on import'),
      '#description' => $this->t('If checked, a new revision will be generated for every node imported from JSON.'),
      '#default_value' => $config->get('import_create_new_revision') ?? FALSE,
    ];

    $form['import_settings']['import_user'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Importer User'),
      '#target_type' => 'user',
      '#default_value' => $config->get('import_user') ? $this->entityTypeManager->getStorage('user')->load($config->get('import_user')) : NULL,
      '#description' => $this->t('Select the user that will be used as author when entities are imported. If you are using workflows or moderation, make sure this user has the necessary permissions. If no user is selected, the entity will be imported as the anonymous user.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('entity_io.import_settings')
      ->set('import_create_new_revision', $form_state->getValue('import_create_new_revision'))
      ->set('import_user', $form_state->getValue('import_user'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
