<?php

namespace Drupal\entity_io\Helper;

use Drupal\path_alias\Entity\PathAlias;

/**
 * Helper class to create entities from a JSON-like array structure.
 */
class EntityPathAlias {

  /**
   * Create or retrieve a URL alias for a single entity using JSON input.
   *
   * @param string $json
   *   JSON string with one object containing:
   *     - entity_id: the ID of the entity (node ID, term ID, etc.)
   *     - alias: the desired URL alias (e.g., "/my-page")
   *     - langcode: optional language code (default "en")
   * @param string $entity_type
   *   Entity type: node, taxonomy_term, user, comment, block_content, media.
   * @param string|null $bundle
   *   Optional bundle (content type for nodes, vocabulary for taxonomy terms).
   * @param string $entity_id
   *   Entity id.
   *
   * @return array|null
   *   Alias info: ['alias', 'source', 'langcode'], or null on error.
   */
  public static function updateAliasFromJson(array $json, string $entity_type, ?string $bundle = NULL, $entity_id = NULL) {
    if (!is_array($json)) {
      return NULL;
    }

    $alias = $json['alias'] ?? '';
    $langcode = $json['langcode'] ?? 'en';

    if (empty($entity_id) || empty($alias)) {
      return NULL;
    }

    $alias_storage = \Drupal::entityTypeManager()->getStorage('path_alias');

    // Determine the system path for the entity.
    switch ($entity_type) {
      case 'node':
        $node = \Drupal::entityTypeManager()->getStorage('node')->load($entity_id);
        if (!$node || ($bundle && $node->bundle() !== $bundle)) {
          return NULL;
        }
        $source = "/node/$entity_id";
        break;

      case 'taxonomy_term':
        $term = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->load($entity_id);
        if (!$term || ($bundle && $term->bundle() !== $bundle)) {
          return NULL;
        }
        $source = "/taxonomy/term/$entity_id";
        break;

      case 'user':
        $user = \Drupal::entityTypeManager()->getStorage('user')->load($entity_id);
        if (!$user) {
          return NULL;
        }
        $source = "/user/$entity_id";
        break;

      case 'comment':
        $comment = \Drupal::entityTypeManager()->getStorage('comment')->load($entity_id);
        if (!$comment) {
          return NULL;
        }
        $source = "/comment/$entity_id";
        break;

      case 'block_content':
        $block = \Drupal::entityTypeManager()->getStorage('block_content')->load($entity_id);
        if (!$block) {
          return NULL;
        }
        $source = "/block_content/$entity_id";
        break;

      case 'media':
        $media = \Drupal::entityTypeManager()->getStorage('media')->load($entity_id);
        if (!$media) {
          return NULL;
        }
        $source = "/media/$entity_id";
        break;

      default:
        // For unknown entities, assume $entity_id is a full system path.
        $source = $entity_id;
        break;
    }

    // Check if an alias exists for this source and language.
    $existing = $alias_storage->loadByProperties([
      'path' => $source,
      'langcode' => $langcode,
    ]);

    if ($existing) {
      /** @var \Drupal\path_alias\Entity\PathAlias $path_alias_entity */
      $path_alias_entity = reset($existing);

      // If the alias is different, update it.
      if ($path_alias_entity->getAlias() !== $alias) {
        $path_alias_entity->set('alias', $alias);
        $path_alias_entity->save();
      }

      return $path_alias_entity;
    }

    // Create new alias if it doesn't exist.
    $path_alias = PathAlias::create([
      'path' => $source,
      'alias' => $alias,
      'langcode' => $langcode,
    ]);
    $path_alias->save();

    return $path_alias;
  }

}
