<?php

namespace Drupal\entity_io\Service;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\entity_io\Helper\EntityCreate;
use Drupal\entity_io\Helper\EntityFields;
use Drupal\entity_io\Helper\EntityLoader;
use Drupal\entity_io\Helper\EntityLogger;
use Drupal\entity_io\Helper\EntityPathAlias;
use Drupal\entity_io\Helper\JsonParseEntityData;
use Drupal\file\Entity\File;
use Drupal\node\Entity\Node;
use Drupal\user\Entity\User;
use Psr\Log\LoggerInterface;

/**
 * Imports entities from JSON data into Drupal.
 */
class EntityImporter {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $fieldManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Global index of imported entities to prevent duplicatesduring.
   *
   * @var array
   */
  protected static $globalEntities = [];

  /**
   * Global index of entity validation messages.
   *
   * @var array
   */
  public static $entityValidation = [];

  /**
   * Exported entities for reference during import.
   *
   * @var array
   */
  public static $exportedEntities = [];

  /**
   * Already imported entities to prevent infinite recursion.
   *
   * @var array
   */
  public static $alreadyImported = [];

  /**
   * Constructs a new EntityImporter object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $fieldManager
   *   The entity field manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    EntityFieldManagerInterface $fieldManager,
    LoggerInterface $logger,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->fieldManager = $fieldManager;
    $this->logger = $logger;
  }

  /**
   * Imports entities from JSON data.
   *
   * @param array $json_
   *   The JSON data to import.
   * @param string|null $langcode
   *   The language code to use.
   * @param bool $debug
   *   Whether to enable debug mode.
   *
   * @return array
   *   The imported entities.
   */
  public function import(array $json_, $langcode = NULL, $debug = FALSE) {

    $config = \Drupal::service('config.factory')->get('entity_io.import_settings');
    if ($config->get('import_user')) {
      // Load the user you want to impersonate.
      // UID 5 for example.
      $user = User::load($config->get('import_user'));
      if ($user instanceof User) {
        // Switch to that user for this scope.
        $account_switcher = \Drupal::service('account_switcher');
        $account_switcher->switchTo($user);
      }
    }

    $entity_values = [];
    if ($json_) {
      foreach ($json_ as $delta => $json) {
        try {
          $json = $this->preventEntityCreationLanguage($json);
          $entity = NULL;
          $type = NULL;
          $id = NULL;
          $uuid = NULL;
          $bundle = NULL;

          if (!$json) {
            continue;
          }

          if (isset($json['exportedEntities'])) {
            self::$exportedEntities = $json['exportedEntities'];
          }

          $entityData = JsonParseEntityData::parseEntityInfo($json);
          extract($entityData);

          if (!$type || !$bundle || !$uuid) {
            throw new \RuntimeException('Missing required keys (entity_type, bundle, uuid).');
          }
          if (!$this->entityTypeManager->hasDefinition($type)) {
            throw new \RuntimeException("Unknown entity type: $type");
          }

          // Load entity.
          $storage = $this->entityTypeManager->getStorage($type);
          $existing = $storage->loadByProperties(['uuid' => $uuid]);
          if (!empty($existing)) {
            $entity = reset($existing);
            $entity = $storage->loadUnchanged($entity->id());
          }
          elseif ($id && ($loaded = $storage->load($id))) {
            $entity = $loaded;
          }

          if (!$langcode && !empty($entity)) {
            $langcode = $entity->langcode->value;
          }
          elseif (!$langcode) {
            $langcode = $json['langcode'][0]['value'];
          }

          // Try to load entity in the json language if exists.
          // It prevents importing an export in a translation language.
          if ($entity && $entity->langcode->value != $json['langcode'][0]['value']) {
            if ($entity->hasTranslation($json['langcode'][0]['value'])) {
              $entity = $entity->getTranslation($json['langcode'][0]['value']);
              $langcode = $entity->langcode->value;
            }
          }

          // If entity already exists, use it to prevent recursion.
          if ($entity && isset(self::$alreadyImported[$uuid]) && in_array($langcode, self::$alreadyImported[$uuid], TRUE)) {
            $entity_values[$delta] = $entity;
            continue;
          }
          self::$alreadyImported[$uuid][] = $langcode;

          $bind = \Drupal::service('entity_io.storage')->get($uuid);
          // Try to get binded entity. It is useful to prevent
          // duplicate entries for the same uuid.
          if (!$entity && $bind) {
            $entity = EntityLoader::loadEntity($bind);
          }

          // Add entity to global index.
          if ($entity) {
            self::$globalEntities[$uuid] = $entity;
          }

          if (isset(self::$globalEntities[$uuid])) {
            $entity = self::$globalEntities[$uuid];
          }
          else {
            $entity = EntityCreate::create($type, $bundle, $langcode, $json);
            if ($entity) {
              self::$globalEntities[$uuid] = $entity;
            }
          }

          if (!$entity) {
            throw new \RuntimeException("Entity not defined bundle: '$bundle', type: '$type' ");
          }

          // Register uuid.
          \Drupal::service('entity_io.storage')->getOrSave(
            $uuid,
            $id,
            $type,
            $bundle,
            $entity->id(),
            $entity->uuid(),
            $entity instanceof File ? 'file' : $entity->getEntityTypeId(),
            $entity->bundle()
          );
          if ($entity && $entity->isTranslatable()) {
            if ($entity->hasTranslation($langcode)) {
              $entity = $entity->getTranslation($langcode);
            }
            else {
              $entity = $entity->addTranslation($langcode);
            }
          }

          // Get field definitions and validate.
          $active_fields = EntityFields::getAll($type, $bundle);

          foreach ($active_fields as $field_name => $field_definition) {
            $field_type = $field_definition->getType();

            if (!isset($json[$field_name])) {
              continue;
            }

            if (in_array($field_name, [
              'id', 'uid', 'fid', 'tid', 'cid', 'mid', 'nid', 'uuid', 'vid', 'changed', 'langcode',
              'revision_id', 'revision_uid', 'content_translation_uid', 'content_translation_changed',
              'revision_timestamp', 'default_langcode',
            ])) {
              continue;
            }

            if (in_array($field_type, [
              'file_uri', 'created', 'language', 'string', 'password', 'string_long', 'metatag_computed',
              'metatag', 'map', 'text', 'text_long', 'text_with_summary', 'integer', 'decimal', 'float',
              'list_string', 'list_integer', 'list_float', 'datetime', 'timestamp', 'boolean', 'email', 'daterange', 'telephone',
            ])) {
              $entity->set($field_name, $json[$field_name]);
            }
            elseif ($field_type == 'uri') {
              $entity->set($field_name, $json[$field_name]);
            }
            elseif ($field_type == 'link') {
              $entity->set($field_name, $this->replaceEntityUri($json[$field_name]));
            }
            elseif ($field_type == 'path') {
              if (isset($json[$field_name][0]['pid']) && isset($json[$field_name][0]['alias']) && isset($json[$field_name][0]['langcode'])) {
                $alias = EntityPathAlias::updateAliasFromJson($json[$field_name][0], $type, $bundle, $entity->id());
                if ($alias) {
                  $entity->set($field_name, $alias->getAlias());
                }
              }
              else {
                $entity->set($field_name, $json[$field_name]);
              }
            }
            elseif (($field_type == 'entity_reference' || $field_type == 'entity_reference_revisions') && isset($json[$field_name][0]['target__uuid'])) {
              // Try to get binded entity. It is useful to prevent
              // duplicate entries for the same uuid.
              $bind_enity = \Drupal::service('entity_io.storage')->get($json[$field_name][0]['target__uuid']);
              if ($bind_enity) {
                $bind_Entity = EntityLoader::loadEntity($bind_enity);
                $entity->set($field_name, $bind_Entity);
              }
            }
            elseif (
                ($field_type == 'entity_reference' || $field_type == 'entity_reference_revisions') &&
                !in_array($field_name, ['type', 'bundle', 'roles', 'entity_id', 'pid'])
            ) {
              if (isset($json[$field_name][$delta]['target_id'])) {
                $entity->set($field_name, $json[$field_name]);
              }
              else {
                if ($json[$field_name]) {
                  $entity->set($field_name, $this->import($json[$field_name], $langcode));
                }
                else {
                  $entity->set($field_name, []);
                }
              }
            }
            // Skipping this set value. The entity is already
            // created by EntityCreate::create.
            elseif (in_array($field_type, ['entity_reference', 'entity_reference_revisions'], TRUE) && $field_name == 'type') {
            }
            elseif (in_array($field_type, ['entity_reference', 'entity_reference_revisions'], TRUE) && $field_name == 'bundle') {
            }
            elseif (in_array($field_type, ['entity_reference', 'entity_reference_revisions'], TRUE) && $field_name == 'roles') {
              $entity->set($field_name, $json[$field_name]);
            }
            elseif (in_array($field_type, ['entity_reference', 'entity_reference_revisions'], TRUE) && $field_name == 'entity_id') {
              if (isset($json[$field_name][0])) {
                $entity_swap = \Drupal::service('entity_io.storage')->getEntity($json[$field_name][0]['target_id'], $json['entity_type'][0]['value']);
                if ($entity_swap) {
                  $entity->set($field_name, $entity_swap['entity_id']);
                }
              }
            }
            elseif (in_array($field_type, ['entity_reference', 'entity_reference_revisions'], TRUE) && $field_name == 'pid') {
              if ($json[$field_name]) {
                $entity_swap = \Drupal::service('entity_io.storage')->getEntity($json[$field_name][0]['target_id'], 'comment');
                if ($entity_swap) {
                  $entity->set($field_name, $entity_swap['entity_id']);
                }
              }
              else {
                $entity->set($field_name, $json[$field_name]);
              }
            }
            elseif ($field_type == 'comment') {
              try {
                $comment_json = [];
                foreach ($json[$field_name] as $value) {
                  $comment_json[] = $this->import([$value], $langcode);
                }
              }
              catch (\Throwable $th) {
                // Debugging code removed.
                // dd("oi", $th, $field_name, $json[$field_name]);.
              }
            }
            elseif ($field_type == 'user') {
              $entity->set($field_name, $this->import($json[$field_name], $langcode));
            }
            elseif (in_array($field_type, ['image', 'file'])) {
              if (isset($json[$field_name][0]['target__uuid'])) {
                $bind_enity = \Drupal::service('entity_io.storage')->get($json[$field_name][0]['target__uuid']);
                // Try to get binded entity. It is useful to prevent duplicate
                // entries for the same uuid.
                if ($bind_enity) {
                  $bind_Entity = EntityLoader::loadEntity($bind_enity);
                  $entity->set($field_name, $bind_Entity);
                }
              }
              elseif ($json[$field_name]) {
                $_entities = $this->import($json[$field_name], $langcode);
                $entity->set($field_name, $_entities);
                $images = $entity->get($field_name);
                foreach ($images as $delta => $image_item) {
                  self::validateOrCreateFileFromJson($image_item->entity, $json[$field_name][$delta]);
                  if (isset($json[$field_name][$delta]['alt'])) {
                    $image_item->set('alt', $json[$field_name][$delta]['alt']);
                  }
                  if (isset($json[$field_name][$delta]['title'])) {
                    $image_item->set('title', $json[$field_name][$delta]['title']);
                  }
                }
              }
            }
            else {
              // Debugging code removed.
              throw "Undefined entity type ->" . $field_type . '<- field_name: ' . $field_name;
            }
          }

          $config = \Drupal::service('config.factory')->get('entity_io.import_settings');
          if ($config->get('import_create_new_revision')) {
            $current_user = \Drupal::currentUser();
            if ($entity instanceof Node && $entity->getEntityType()->isRevisionable()) {
              // Create a new revision.
              $entity->setNewRevision(TRUE);
              $entity->setRevisionCreationTime(\Drupal::time()->getRequestTime());
              $entity->setRevisionUserId($current_user->id());
              $entity->setRevisionLogMessage('{entity_io:IMPORT} Revision created by user: ' . $current_user->getAccountName());
            }
          }

          if ($entity instanceof Node && $entity->hasField('moderation_state')) {
            $current_state = $entity->get('moderation_state')->value;
            $new_state = $json['moderation_state'][0]['value'] ?? NULL;

            if ($new_state && $new_state !== $current_state) {
              /** @var \Drupal\content_moderation\ModerationInformation $moderation_info */
              $moderation_info = \Drupal::service('content_moderation.moderation_information');
              $workflow = $moderation_info->getWorkflowForEntity($entity);

              if ($workflow) {
                $state = $workflow->getTypePlugin()->getState($current_state);

                // Transições possíveis a partir do estado atual.
                $transitions = $state->getTransitions();

                $valid = FALSE;
                foreach ($transitions as $transition) {
                  if ($transition->getToState() === $new_state) {
                    $valid = TRUE;
                    break;
                  }
                }

                if ($valid) {
                  $entity->set('moderation_state', $new_state);
                }
                else {
                  // Log invalid state change attempt.
                  \Drupal::logger('entity_io')->warning(
                    "Tentativa de mudar de @from para @to inválida para node @nid",
                    ['@from' => $current_state, '@to' => $new_state, '@nid' => $entity->id()]
                  );
                }
              }
            }
          }

          // Validate entity before save.
          // It can be useful to log validation errors.
          $violations = $entity->validate();
          if ($violations->count() === 0) {
            // \Drupal::logger('validation')->info('Node is valid!');
          }
          else {
            foreach ($violations as $violation) {
              self::$entityValidation[] = EntityLogger::getEntityTitle($entity) . '->' . $violation->getPropertyPath() . ': ' . $violation->getMessage();
            }
          }

          \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
          $entity->save();
          \Drupal::state()->delete('entity_io.skip_webhooks');
          $entity = EntityLoader::reloadEntity($entity);

          if ($entity->isTranslatable()) {
            $languages = \Drupal::languageManager()->getLanguages();
            foreach ($languages as $langid => $language) {
              if ($entity->langcode->value == $langid) {
                continue;
              }

              if (isset($json['translations']) && in_array($langid, $json['translations'])) {
                if (isset(self::$exportedEntities[$json['uuid'][0]['value']][$langid])) {
                  if (self::$exportedEntities[$json['uuid'][0]['value']][$langid]) {
                    $translated_entity = $this->import([self::$exportedEntities[$json['uuid'][0]['value']][$langid]], $langid);
                    foreach ($translated_entity as $e) {
                      \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
                      $e->save();
                      \Drupal::state()->delete('entity_io.skip_webhooks');
                    }
                  }
                }
              }
            }
          }

          $entity_values[$delta] = $entity;
        }
        catch (\Throwable $th) {
          \Drupal::service('entity_io.logger')->log('import', $type, NULL, 'error', $th->getMessage());
          throw $th;
        }
      }
    }

    \Drupal::service('entity_io.logger')->log('import', $type, $entity->id(), 'success', 'Import successful.');
    return $entity_values;
  }

  /**
   * Prevents entity creation in a non-default language.
   *
   * @param array $json
   *   The JSON data.
   *
   * @return array
   *   The possibly modified JSON data.
   */
  protected function preventEntityCreationLanguage($json) {
    $default_langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();

    if (isset($json['translations']) && isset($json['langcode'][0]['value']) && isset($json['translations'][$default_langcode]) && $json['langcode'][0]['value'] != $default_langcode) {
      $new_json = [];
      $translations = [];

      if ($json['langcode'][0]['value'] != $default_langcode) {
        foreach ($json['translations'] as $key => $value) {
          $translations[$key] = $value;
        }

        unset($json['translations']);

        $translations[$json['langcode'][0]['value']] = $json;

        $new_json = $translations[$default_langcode];
        unset($translations[$default_langcode]);
        $new_json['translations'] = $translations;
      }
    }

    return $new_json ?? $json;
  }

  /**
   * Gets the required fields for an entity type and bundle.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The bundle.
   *
   * @return array
   *   The required field names.
   */
  public static function getMandatoryFields($type, $bundle) {
    $field_definitions = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions($type, $bundle);

    $required_fields = [];

    foreach ($field_definitions as $field_name => $definition) {
      if (!$definition->isComputed() && $definition->isRequired()) {
        $required_fields[] = $field_name;
      }
    }

    return $required_fields;
  }

  /**
   * Validates if the file attached to an entity matches the JSON.
   *
   * If not, creates a new file using base64 data from the JSON.
   *
   * @param \Drupal\file\Entity\File $entity
   *   The entity containing the file field.
   * @param array $json
   *   The JSON data. Must include:
   *     - filename: The file name.
   *     - uri: Destination URI (e.g., 'public://myfolder/file.jpg').
   *     - filemime: MIME type (e.g., 'image/jpeg').
   *     - base64: Base64-encoded file content.
   *     - field_name: Name of the file field in the entity.
   *
   * @return \Drupal\file\Entity\File|false
   *   The valid file entity, or FALSE if invalid or failed to create.
   */
  public static function validateOrCreateFileFromJson(File $entity, array $json): File|false {
    if (!isset($json['filename']) || !$entity->get('filename')) {
      return FALSE;
    }

    $is_match = $entity instanceof File
      && $entity->getFilename() === $json['filename'][0]['value']
      && $entity->getFileUri() === $json['uri'][0]['value']
      && $entity->getMimeType() === $json['filemime'][0]['value']
      && file_exists($json['uri'][0]['value'])
      && file_get_contents($entity->getFileUri()) == $json['base64'];

    if ($is_match) {
      return $entity;
    }

    // Validate required keys for creation.
    if (empty($json['base64']) || empty($json['filename'][0]['value']) || empty($json['uri'][0]['value']) || empty($json['filemime'][0]['value'])) {
      return FALSE;
    }

    // Prepare the destination directory.
    $directory = dirname($json['uri'][0]['value']);
    \Drupal::service('file_system')->prepareDirectory(
      $directory,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    // Decode base64.
    $data = base64_decode($json['base64']);
    if ($data === FALSE) {
      return FALSE;
    }

    $destination = $json['uri'][0]['value'];

    // Save the file in the system and set it as permanent.
    $file = \Drupal::service('file.repository')->writeData(
      $data,
      $destination,
      FileSystemInterface::EXISTS_REPLACE
    );
    $file->setMimeType($json['filemime'][0]['value']);
    $file->setFilename($json['filename'][0]['value']);
    // $file->setPermanent();
    $file->save();

    return $file;
  }

  /**
   * Replaces the entity URI in a value array with the correct entity.
   *
   * @param array $value
   *   The value array containing 'target__uuid' and 'uri'.
   *
   * @return array
   *   The value array with the updated 'uri' if the entity is found.
   */
  public function replaceEntityUri($value) {
    if (isset($value['target__uuid']) && isset($value['uri']) && preg_match('/^entity:([a-z_]+)\/(\d+)$/', $value['uri'], $matches)) {
      // "node"
      $entity_type = $matches[1];
      // 25
      $target__uuid = $value['target__uuid'];

      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->loadByProperties(['uuid' => $target__uuid]);
      if ($entity) {
        $entity = reset($entity);
        $value['uri'] = "entity:{$entity_type}/{$entity->id()}";
        return $value;
      }
      else {
        $bind = \Drupal::service('entity_io.storage')->get($target__uuid);
        // Try to get binded entity.
        if (!$entity && $bind) {
          $entity = EntityLoader::loadEntity($bind);
          if ($entity) {
            $value['uri'] = "entity:{$entity_type}/{$entity->id()}";
            return $value;
          }
        }
      }
    }
    return $value;
  }

}
