<?php

namespace Drupal\entity_io_queue\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\entity_io\Service\EntityIoExporter;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes entity export queue items.
 *
 * @QueueWorker(
 *   id = "entity_io_queue_export",
 *   title = @Translation("Entity IO Export Queue Worker"),
 *   cron = {"time" = 60}
 * )
 */
class EntityIoExportQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new EntityIoExportQueueWorker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entityTypeManager, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('logger.factory')->get('entity_io')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    try {
      $entity_type = $data['entity_type'];
      $entity_id = $data['entity_id'];
      $depth = $data['depth'] ?? 0;
      $max_depth = $data['max_depth'] ?? 3;
      $selected_fields = $data['selected_fields'] ?? [];
      $langcode = $data['langcode'] ?? 'en';
      $is_revision = $data['is_revision'] ?? FALSE;

      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->load($entity_id);

      if (!$entity) {
        throw new \Exception("Entity {$entity_type}:{$entity_id} not found");
      }

      $url = EntityIoExporter::toJson($entity, $depth, $max_depth, $selected_fields, $langcode, $is_revision);

      $this->logger->info('Entity @type:@id exported to @url', [
        '@type' => $entity_type,
        '@id' => $entity_id,
        '@url' => $url,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to export entity @type:@id: @message', [
        '@type' => $data['entity_type'] ?? 'unknown',
        '@id' => $data['entity_id'] ?? 'unknown',
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

}
