<?php

namespace Drupal\entity_io\Form\FormSettings;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\entity_io\Helper\EntityFields;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Config form to select fields for File entities.
 */
class FileFieldConfigForm extends FormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs the form.
   */
  public function __construct(ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, ConfigFactoryInterface $config_factory) {
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_file_field_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['entity_io.file_field_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('entity_io.file_field_settings');

    if (!$this->moduleHandler->moduleExists('file')) {
      $form['message'] = [
        '#markup' => $this->t('<p><strong>The File module is not enabled.</strong></p><p>Please enable the File module to configure file fields.</p>'),
      ];
      return $form;
    }

    $form['entity_base_fields']['#markup'] = Markup::create('
      <div style="background: #f9f9f9; padding: 12px 16px; border-left: 4px solid #f39c12; border-radius: 4px; margin-bottom: 12px;">
        <p style="margin: 0 0 6px; font-weight: 600; color: #444;">⚙ Flexible export</p>
        <p style="margin: 4px 0;">This module allows you to choose exactly which fields to include in the export, giving you full flexibility — you can export only the information you need.</p>
        <p style="margin: 8px 0 0; color: #555;">However, if your goal is to <strong>import the exported JSON</strong> later, you must include <strong>all Base Fields</strong> in the export to ensure compatibility.</p>
      </div>
    ');

    // File entity has no bundles; use 'file' as the config key.
    $bundle = 'file';

    // Base fields available on the file entity.
    $base_fields = EntityFields::getBaseFields('file');

    $form[$bundle] = [
      '#type' => 'details',
      '#title' => $this->t('File Fields'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form[$bundle]['base_fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Base Fields'),
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      // Skip computed/internal fields if desired.
      $default = $config->get($bundle)['base_fields'][$base_field_name] ?? '';
      $form[$bundle]['base_fields'][$base_field_name] = [
        '#type' => 'checkbox',
        '#title' => $base_field_name,
        '#default_value' => $default,
      ];
    }

    // Custom field_config fields for file entity (if any).
    $fields = EntityFields::get('file', 'file');

    if ($fields) {
      $form[$bundle]['fields'] = [
        '#type' => 'details',
        '#title' => $this->t('Fields'),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      foreach ($fields as $field) {
        $default = $config->get($bundle)['fields'][$field->getName()] ?? '';
        $form[$bundle]['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->label() . ' (' . $field->getName() . ')',
          '#default_value' => $default,
        ];
      }
    }

    // Depth select.
    $options = [];
    for ($i = -1; $i <= 100; $i++) {
      $options[(string) $i] = (string) $i;
    }

    $form[$bundle]['depth'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity reference depth'),
      '#options' => $options,
      '#default_value' => $config->get($bundle)['depth'] ?? -1,
      '#description' => $this->t('Depth level for exporting referenced entities. -1 = unlimited, 0 = none, 1 = direct references only.'),
    ];

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => $this->t('Check All'),
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => $this->t('Clear All'),
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => $this->t('Toggle'),
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $editable = $this->configFactory->getEditable('entity_io.file_field_settings');

    foreach ($values as $key => $value) {
      if (in_array($key, [
        'submit',
        'form_token',
        '_token',
        'form_build_id',
        'form_id',
        'op',
        'check_all',
        'clear_all',
        'toggle',
      ])) {
        continue;
      }

      if (!is_array($value)) {
        continue;
      }

      $data = [];
      if (isset($value['fields'])) {
        $data['fields'] = array_filter($value['fields']);
      }
      if (isset($value['base_fields'])) {
        $data['base_fields'] = array_filter($value['base_fields']);
      }
      if (isset($value['depth'])) {
        $data['depth'] = $value['depth'];
      }

      $editable->set($key, $data);
    }

    $editable->save();
  }

}
