<?php

namespace Drupal\entity_io\Helper;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Helper to manage export directory paths and files.
 *
 * Provides utilities to load configuration, list, save and delete export files
 * stored under the configured stream wrapper (public:// or private://).
 */
class ExportDirectory {

  /**
   * Stream wrapper scheme (e.g. 'public' or 'private').
   *
   * @var string
   */
  public static $scheme;

  /**
   * Base directory configured for exports (without scheme).
   *
   * @var string
   */
  public static $directory;

  /**
   * Stream wrapper base path (e.g. public://entity_io_exports).
   *
   * @var string
   */
  public static $publicPath;

  /**
   * Resolved real filesystem path for the publicPath.
   *
   * @var string|null
   */
  public static $publicRealPath;

  /**
   * Stream wrapper path for the last saved file.
   *
   * @var string|null
   */
  public static $publicPathFile;

  /**
   * Real filesystem path for the last saved file.
   *
   * @var string|null
   */
  public static $publicRealPathFile;

  /**
   * Public URL for the last saved file (if applicable).
   *
   * @var string|null
   */
  public static $publicUrl;

  /**
   * Generated file name for the last saved file.
   *
   * @var string|null
   */
  public static $fileName;

  /**
   * Load configuration and initialize paths.
   */
  public static function loadConfig() {
    $config = \Drupal::config('entity_io.export_settings');
    $scheme = $config->get('storage_scheme') ?? 'public';
    $directory = $config->get('directory') ?? 'entity_io_exports';

    self::$scheme = $scheme;
    self::$directory = $directory;
    self::$publicPath = $scheme . '://' . $directory;

    $file_system = \Drupal::service('file_system');
    self::$publicRealPath = $file_system->realpath(self::$publicPath);
  }

  /**
   * Get files from export directory.
   *
   * If $filter is '*', return all files recursively from all subdirectories.
   * If $filter is an entity type, return files only from that subdirectory.
   *
   * @param string $filter
   *   The '*' or an entity type folder name.
   *
   * @return array
   *   Array of stream wrapper paths.
   */
  public static function getFiles($filter = '*'): array {
    self::loadConfig();

    $files = [];
    $directory = self::$publicRealPath;

    if (!is_dir($directory)) {
      return $files;
    }

    if ($filter === '*') {
      $files = self::getFilesRecursively($directory);
    }
    else {
      $entity_type_directory = $directory . '/' . $filter;
      if (is_dir($entity_type_directory)) {
        $files = self::getFilesFromDirectory($entity_type_directory, $filter . '/');
      }
    }

    return $files;
  }

  /**
   * Get all files recursively from directory.
   */
  private static function getFilesRecursively($directory, $prefix = ''): array {
    $files = [];

    if ($handle = opendir($directory)) {
      while (FALSE !== ($entry = readdir($handle))) {
        if ($entry === '.' || $entry === '..') {
          continue;
        }

        $full_path = $directory . '/' . $entry;
        $relative_path = $prefix . $entry;

        if (is_dir($full_path)) {
          // Recursively get files from subdirectory.
          $subfiles = self::getFilesRecursively($full_path, $relative_path . '/');
          $files = array_merge($files, $subfiles);
        }
        else {
          // It's a file - return full stream wrapper path.
          $stream_path = self::$publicPath . '/' . $relative_path;
          $files[] = $stream_path;
        }
      }
      closedir($handle);
    }

    return $files;
  }

  /**
   * Get files from a specific directory.
   */
  private static function getFilesFromDirectory($directory, $prefix = ''): array {
    $files = [];

    if ($handle = opendir($directory)) {
      while (FALSE !== ($entry = readdir($handle))) {
        if ($entry === '.' || $entry === '..') {
          continue;
        }

        $full_path = $directory . '/' . $entry;

        if (is_file($full_path)) {
          $relative_path = $prefix . $entry;
          $stream_path = self::$publicPath . '/' . $relative_path;
          $files[] = $stream_path;
        }
      }
      closedir($handle);
    }

    return $files;
  }

  /**
   * Delete files from a specific directory.
   */
  private static function deleteFilesFromDirectory($directory, $prefix = ''): array {
    $deleted_files = [];
    $file_system = \Drupal::service('file_system');

    if ($handle = opendir($directory)) {
      while (FALSE !== ($entry = readdir($handle))) {
        if ($entry === '.' || $entry === '..') {
          continue;
        }

        $full_path = $directory . '/' . $entry;

        if (is_file($full_path)) {
          // Ensure file is writable before deletion.
          if (!is_writable($full_path)) {
            @chmod($full_path, 0666);
          }
          // Convert to stream wrapper URI for Drupal file system.
          $base_directory = self::$publicRealPath;
          $relative_path = str_replace($base_directory . '/', '', $full_path);
          $stream_uri = self::$publicPath . '/' . $relative_path;

          try {
            if ($file_system->delete($stream_uri)) {
              $deleted_files[] = $stream_uri;
            }
            else {
              // Log warning if file could not be deleted.
              \Drupal::logger('entity_io_purge')->warning('Permission denied deleting file: @path', ['@path' => $full_path]);
            }
          }
          catch (\Exception $e) {
            \Drupal::logger('entity_io_purge')->error('Failed to delete file @uri: @error', [
              '@uri' => $stream_uri,
              '@error' => $e->getMessage(),
            ]);
          }
        }
      }
      closedir($handle);
    }

    return $deleted_files;
  }

  /**
   * Clear exported files.
   *
   * @param string|null $entity_type
   *   Optional entity type to filter deletions. If NULL, all files are deleted.
   *
   * @return array
   *   Array with 'deleted_count' and 'deleted_files' keys.
   */
  public static function clearFiles($entity_type = NULL): array {
    self::loadConfig();

    $deleted_files = [];
    $directory = self::$publicRealPath;

    if (!is_dir($directory)) {
      return ['deleted_count' => 0, 'deleted_files' => []];
    }

    if ($entity_type) {
      $entity_type_directory = $directory . '/' . $entity_type;
      if (is_dir($entity_type_directory)) {
        $deleted_files = self::deleteFilesFromDirectory($entity_type_directory);
      }
    }
    else {
      $deleted_files = self::deleteFilesRecursively($directory);
    }

    return [
      'deleted_count' => count($deleted_files),
      'deleted_files' => $deleted_files,
    ];
  }

  /**
   * Delete all files recursively from directory.
   */
  private static function deleteFilesRecursively($directory, $prefix = ''): array {
    $deleted_files = [];
    $file_system = \Drupal::service('file_system');

    if ($handle = opendir($directory)) {
      while (FALSE !== ($entry = readdir($handle))) {
        if ($entry === '.' || $entry === '..') {
          continue;
        }

        $full_path = $directory . '/' . $entry;
        $relative_path = $prefix . $entry;

        if (is_dir($full_path)) {
          // Recursively delete files from subdirectory.
          $subfiles = self::deleteFilesRecursively($full_path, $relative_path . '/');
          $deleted_files = array_merge($deleted_files, $subfiles);
        }
        else {
          // It's a file - delete it.
          try {
            // Convert to stream wrapper URI for Drupal file system.
            $stream_uri = self::$publicPath . '/' . $relative_path;

            if ($file_system->delete($stream_uri)) {
              $deleted_files[] = $stream_uri;
            }
          }
          catch (\Exception $e) {
            \Drupal::logger('entity_io_purge')->error('Failed to delete file @uri: @error', [
              '@uri' => self::$publicPath . '/' . $relative_path,
              '@error' => $e->getMessage(),
            ]);
          }
        }
      }
      closedir($handle);
    }

    return $deleted_files;
  }

  /**
   * Check if directory is empty.
   */
  private static function isDirectoryEmpty($directory): bool {
    $handle = opendir($directory);
    while (FALSE !== ($entry = readdir($handle))) {
      if ($entry != '.' && $entry != '..') {
        closedir($handle);
        return FALSE;
      }
    }
    closedir($handle);
    return TRUE;
  }

  /**
   * Prepare directory structure for entity export.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to prepare directory for.
   * @param string $filename
   *   The filename to be saved.
   *
   * @return array
   *   Array with directory paths information.
   */
  public static function prepareExportDirectory(EntityInterface $entity, string $filename): array {
    self::loadConfig();

    $entity_type = $entity->getEntityTypeId();

    // Prepare directory paths.
    $entity_directory = self::$publicPath . '/' . $entity_type;
    $entity_real_directory = self::$publicRealPath . '/' . $entity_type;
    $file_path = $entity_directory . '/' . $filename;
    $real_file_path = $entity_real_directory . '/' . $filename;

    // Create directory if it doesn't exist.
    $file_system = \Drupal::service('file_system');
    if (!is_dir($entity_real_directory)) {
      $file_system->prepareDirectory($entity_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    }

    return [
      'entity_type' => $entity_type,
      'directory' => $entity_directory,
      'real_directory' => $entity_real_directory,
      'file_path' => $file_path,
      'real_file_path' => $real_file_path,
      'filename' => $filename,
    ];
  }

  /**
   * Save JSON file to export directory.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being exported.
   * @param string $json
   *   The JSON content to save.
   * @param string $filename
   *   The filename to save.
   *
   * @return array
   *   Array with save result and file information.
   */
  public static function saveJsonFile(EntityInterface $entity, string $json, string $filename): array {
    $directory_info = self::prepareExportDirectory($entity, $filename);

    $bytes_written = file_put_contents($directory_info['real_file_path'], $json);

    if ($bytes_written === FALSE) {
      return [
        'success' => FALSE,
        'error' => 'Failed to write file',
        'file_path' => $directory_info['file_path'],
        'bytes_written' => 0,
      ];
    }

    // Update last-file properties.
    self::$publicPathFile = $directory_info['file_path'];
    self::$publicRealPathFile = $directory_info['real_file_path'];
    self::$fileName = $filename;

    return [
      'success' => TRUE,
      'file_path' => $directory_info['file_path'],
      'real_file_path' => $directory_info['real_file_path'],
      'bytes_written' => $bytes_written,
      'entity_type' => $directory_info['entity_type'],
      'filename' => $filename,
    ];
  }

  /**
   * Return total size (in bytes) of the configured export directory.
   *
   * Uses self::loadConfig() to initialize the configured stream wrapper path
   * and resolves it to a real filesystem path before summing file sizes.
   *
   * @return int
   *   Total size in bytes. Returns 0 if directory does not exist or on error.
   */
  public static function getDirectorySize(): int {
    // Ensure configuration and paths are loaded.
    self::loadConfig();

    $directory = self::$publicRealPath;
    if (empty($directory) || !is_dir($directory)) {
      return 0;
    }

    $size = 0;

    try {
      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($directory, \FilesystemIterator::SKIP_DOTS)
      );

      foreach ($iterator as $file) {
        /** @var \SplFileInfo $file */
        if (!$file->isFile()) {
          continue;
        }
        $filesize = $file->getSize();
        if ($filesize !== FALSE) {
          $size += $filesize;
        }
      }
    }
    catch (\Throwable $e) {
      // Log and return what we've accumulated so far (or 0).
      if (\Drupal::hasService('logger.channel.entity_io')) {
        \Drupal::logger('entity_io')->warning('Failed to calculate export directory size: @msg', ['@msg' => $e->getMessage()]);
      }
      return (int) $size;
    }

    return (int) $size;
  }

}
