<?php

namespace Drupal\entity_io_purge\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\entity_io_purge\Service\PurgeService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for manual purge actions.
 */
class PurgeController extends ControllerBase {

  /**
   * The purge service.
   *
   * @var \Drupal\entity_io_purge\Service\PurgeService
   */
  protected PurgeService $purgeService;

  public function __construct(PurgeService $purgeService) {
    $this->purgeService = $purgeService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_io_purge.service')
    );
  }

  /**
   * Show purge selection form.
   */
  public function purgeForm() {
    return $this->formBuilder()->getForm('Drupal\entity_io_purge\Form\PurgeSelectionForm');
  }

  /**
   * Show confirmation form before purging.
   */
  public function purgeConfirm(Request $request) {
    $entity_types = $request->query->all()['entity_types'] ?? [];

    if (empty($entity_types)) {
      return new RedirectResponse('/admin/config/entity-io/purge');
    }

    return $this->formBuilder()->getForm('Drupal\entity_io_purge\Form\PurgeConfirmForm', $entity_types);
  }

  /**
   * Execute the purge operation.
   */
  public function purge(Request $request) {
    $entity_types = $request->query->all()['entity_types'] ?? [];

    if (empty($entity_types)) {
      $this->messenger()->addError($this->t('No entity types selected.'));
      return new RedirectResponse('/admin/config/entity-io/purge');
    }

    // Use manual purge method that bypasses frequency checks.
    $results = $this->purgeService->purgeByEntityTypes($entity_types);

    $total_deleted = 0;
    foreach ($results as $entity_type => $result) {
      $total_deleted += $result['deleted_count'];
      if ($result['deleted_count'] > 0) {
        $this->messenger()->addStatus($this->t('Deleted @count files for @type entities.', [
          '@count' => $result['deleted_count'],
          '@type' => $entity_type,
        ]));
      }
      else {
        $this->messenger()->addWarning($this->t('No files found for @type entities.', [
          '@type' => $entity_type,
        ]));
      }
    }

    if ($total_deleted > 0) {
      $this->messenger()->addStatus($this->t('Cleanup executed successfully. Total @count files deleted.', [
        '@count' => $total_deleted,
      ]));
    }
    else {
      $this->messenger()->addWarning($this->t('No files were deleted.'));
    }

    return new RedirectResponse('/admin/config/entity-io/purge');
  }

}
