<?php

namespace Drupal\entity_io_push\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\entity_io\Helper\JsonParseEntityData;
use Drupal\entity_io\Helper\JsonValidate;
use Drupal\entity_io\Service\EntityImporter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller to receive JSON and return a response.
 */
class JsonReceiverController extends ControllerBase {

  /**
   * The entity importer service.
   *
   * @var \Drupal\entity_io\Service\EntityImporter
   */
  protected $entityImporter;

  /**
   * Constructs a new JsonReceiverController object.
   *
   * @param \Drupal\entity_io\Service\EntityImporter $entity_importer
   *   The entity importer service.
   */
  public function __construct(EntityImporter $entity_importer) {
    $this->entityImporter = $entity_importer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_io.entity_importer')
    );
  }

  /**
   * Receives JSON and returns a status.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with status.
   */
  public function receive(Request $request): JsonResponse {
    $json_data = json_decode($request->getContent(), TRUE);

    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid JSON payload.',
        'error' => json_last_error_msg(),
      ], 400);
    }

    $json_data = $json_data[0];
    $jsonValidation = JsonValidate::validateEntity($json_data);
    $jsonLogs = JsonValidate::getLog();
    $jsonErrors = JsonValidate::getErrors();

    if (!$jsonValidation) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid JSON data',
        'logs' => $jsonLogs,
        'error' => $jsonErrors,
      ], 400);
    }

    $entityData = JsonParseEntityData::parseEntityInfo($json_data);
    extract($entityData);

    // First validation to know if the entity
    // is valid and have the mandatory fields.
    if (!$type || !$bundle || !$uuid) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid JSON entity. Type, bundle, uuid required!',
        'logs' => $jsonLogs,
        'error' => $jsonErrors,
      ], 400);
    }

    try {
      $entities = $this->entityImporter->import([$json_data]);

      if (!empty($entities)) {
        $entity = reset($entities);

        // Generate the absolute URL of the imported
        // entity if it supports canonical route.
        $entity_url = '';
        if ($entity->hasLinkTemplate('canonical')) {
          $entity_url = $entity->toUrl('canonical', ['absolute' => TRUE])->toString();
        }

        return new JsonResponse([
          'success' => TRUE,
          'message' => 'OK',
          'entity_url' => $entity_url,
          'entity_type' => $entity->getEntityTypeId(),
          'entity_id' => $entity->id(),
        ], 200);
      }
    }
    catch (\Throwable $th) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Importing error.',
        'error' => $th->getMessage(),
      ], 400);
    }

    return new JsonResponse([
      'success' => FALSE,
      'message' => 'Unexpected error.',
    ], 400);
  }

}
