<?php

namespace Drupal\entity_io_push\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\entity_io_push\Service\JsonPushService;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to deploy an entity JSON to a selected server.
 */
class DeployEntityTaxonomyForm extends FormBase {

  /**
   * The JsonPushService.
   *
   * @var \Drupal\entity_io_push\Service\JsonPushService
   */
  protected $pushService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The field config storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $fieldConfigStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity IO export service.
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The entity being deployed.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $entity;

  /**
   * Constructs a DeployEntityTaxonomyForm object.
   *
   * @param \Drupal\entity_io_push\Service\JsonPushService $push_service
   *   The JsonPushService service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\entity_io\Service\EntityIoExport $entity_io_export
   *   The entity IO export service.
   */
  public function __construct(
    JsonPushService $push_service,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    LanguageManagerInterface $language_manager,
    $entity_io_export,
  ) {
    $this->pushService = $push_service;
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->fieldConfigStorage = $entity_type_manager->getStorage('field_config');
    $this->languageManager = $language_manager;
    $this->entityIoExport = $entity_io_export;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_io_push.json_push_service'),
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('language_manager'),
      $container->get('entity_io.export')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_io_push_deploy_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
    ?EntityInterface $node = NULL,
    ?EntityInterface $comment = NULL,
    ?EntityInterface $taxonomy_term = NULL,
    ?EntityInterface $user = NULL,
    ?EntityInterface $media = NULL,
    ?EntityInterface $block_content = NULL,
  ) {
    $this->entity = $node ?: $comment ?: $taxonomy_term ?: $user ?: $media ?: $block_content;

    if (!$this->entity) {
      $form['message'] = [
        '#markup' => $this->t('No entity found.'),
      ];
      return $form;
    }

    $servers = $this->configFactory->get('entity_io_push.settings')->get('servers') ?? [];

    if (empty($servers)) {
      $url = Url::fromRoute('entity_io_push.settings');
      $link = Link::fromTextAndUrl($this->t('configure them here'), $url)->toString();

      $form['message'] = [
        '#markup' => $this->t('No servers are configured. Please @link first.', ['@link' => $link]),
      ];

      return $form;
    }

    $options = [];
    foreach ($servers as $key => $server) {
      $options[$key] = $server['name'] . ' (' . $server['url'] . ')';
    }

    $form['server'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Server'),
      '#options' => $options,
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Push'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'deploy-result',
      ],
    ];

    $config = $this->configFactory->get('entity_io.taxonomy_term_field_settings');

    // Get base fields.
    $entity_type = $this->entityTypeManager->getDefinition('taxonomy_term');
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('taxonomy_term', $entity_type->id());

    $form[$this->entity->bundle()]['base_fields'] = [
      '#type' => 'details',
      '#title' => 'Base Fields',
      '#tree' => TRUE,
    ];

    foreach ($field_definitions as $base_field_name => $fieldDefinition) {
      if (!str_starts_with($base_field_name, 'field_')) {
        $default = $config->get($this->entity->bundle())['base_fields'][$base_field_name] ?? '';
        $form[$this->entity->bundle()]['base_fields'][$base_field_name] = [
          '#type' => 'checkbox',
          '#title' => $base_field_name,
          '#default_value' => $default,
        ];
      }
    }

    // Load all fields for this taxonomy vocabulary.
    $fields = $this->fieldConfigStorage->loadByProperties([
      'entity_type' => 'taxonomy_term',
      'bundle' => $this->entity->bundle(),
    ]);

    if ($fields) {
      // Custom fields.
      $form['fields'] = [
        '#type' => 'details',
        '#title' => 'Fields',
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      foreach ($fields as $field) {
        $default = $config->get($this->entity->bundle())['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->getName(),
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => 'Check All',
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => 'Clear All',
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => 'Toggle',
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $depth_options = [];
    for ($i = -1; $i <= 100; $i++) {
      $depth_options[(string) $i] = (string) $i;
    }

    $form['depth'] = [
      '#type' => 'select',
      '#title' => 'Entity reference depth',
      '#options' => $depth_options,
      '#default_value' => -1,
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'deploy-result'],
    ];

    return $form;
  }

  /**
   * Ajax callback for the submit button.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): array {
    return $form['result'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $server_key = $form_state->getValue('server');

    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $max_depth = (int) $form_state->getValue('depth');
    $term = $this->getRequest()->attributes->get('taxonomy_term');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));
    $response = NULL;

    if ($term instanceof Term) {
      $data = $this->entityIoExport::export($term, 0, $max_depth, $selected_fields, $current_langcode);
      $json = json_encode([$data]);
      $response = $this->pushService->push($json, $server_key);
    }

    $message = $response && $response['success']
      ? $this->t('Successfully deployed entity. Server response: @response', ['@response' => $response['body'] ?? 'No response'])
      : $this->t('Deployment failed: @message', ['@message' => $response['message'] ?? 'Unknown error']);

    $this->messenger->addMessage($message);
  }

}
