<?php

namespace Drupal\entity_io\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\entity_io\Helper\ExportDirectory;
use Drupal\entity_io\Service\EntityIoExport;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Question\ConfirmationQuestion;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for Entity IO module.
 */
class EntityManagementCommands extends DrushCommands {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The account switcher service.
   *
   * @var \Drupal\Core\Session\AccountSwitcherInterface
   */
  protected $accountSwitcher;

  /**
   * The entity importer service.
   *
   * @var \Drupal\entity_io\Service\EntityIoImporter
   */
  protected $entityImporter;

  /**
   * The entity_io.export service (concrete).
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var \Drupal\entity_io\Service\EntityIoExporter
   */
  protected $exporter;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs the command object with injected services.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, AccountSwitcherInterface $account_switcher, $entity_importer, EntityIoExport $entity_io_export, $exporter, FileSystemInterface $file_system) {
    parent::__construct();
    $this->entityTypeManager = $entity_type_manager;
    $this->accountSwitcher = $account_switcher;
    $this->entityImporter = $entity_importer;
    $this->entityIoExport = $entity_io_export;
    $this->exporter = $exporter;
    $this->fileSystem = $file_system;
  }

  /**
   * Create from container.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('account_switcher'),
      $container->get('entity_io.entity_importer'),
      $container->get('entity_io.export'),
      $container->get('entity_io.exporter'),
      $container->get('file_system')
    );
  }

  /**
   * Export sample entities or all entities and save JSON files.
   *
   * @command entity-io:export
   * @aliases eioe
   *
   * @usage drush entity-io:export --all
   * @usage drush entity-io:export --entity-type=node
   * @usage drush entity-io:export --entity-type=node --all
   * @usage drush entity-io:export --entity-type=node --id=123
   * @usage drush eioe --all
   *
   * @option all If set, exports all entities instead of just samples.
   * @option entity-type Filter by entity type (node, user, taxonomy_term, etc).
   * @option id Export specific entity by ID (requires entity-type).
   */
  public function export(array $options = ['all' => FALSE, 'entity-type' => NULL, 'id' => NULL]) {
    $this->output()->writeln("<info>Starting export...</info>");
    $exported_files = [];

    // Validate options.
    if (!empty($options['id']) && empty($options['entity-type'])) {
      $this->output()->writeln("<error>When using --id, you must also specify --entity-type</error>");
      return;
    }

    // Check Drush options.
    $all = !empty($options['all']);
    $entity_type = $options['entity-type'];
    $entity_id = $options['id'];

    $entities_to_export = $this->collectEntities($all, $entity_type, $entity_id);

    if (empty($entities_to_export)) {
      $this->output()->writeln("<comment>No entities found to export.</comment>");
      return;
    }

    $this->output()->writeln("<info>Exporting " . count($entities_to_export) . " entities...</info>");

    $progress = new ProgressBar($this->output(), count($entities_to_export));

    foreach ($entities_to_export as $entity) {
      $this->saveExport($entity, $exported_files);
      $progress->advance();
    }

    $progress->finish();
    $this->output()->writeln("\n<info>Export completed.</info>");
    $this->output()->writeln("<info>Exported files:</info>");
    foreach ($exported_files as $file) {
      $this->output()->writeln($file);
    }
  }

  /**
   * Import all exported JSON entities.
   *
   * @command entity-io:import
   * @aliases eioi
   *
   * @usage drush entity-io:import
   * @usage drush eioi
   */
  public function import(array $options = ['user' => NULL, 'entity-type' => NULL]) {
    $switcher = $this->accountSwitcher;

    $run_as_user = NULL;

    if (!empty($options['user'])) {
      $input = $options['user'];

      if (is_numeric($input)) {
        // Load by UID.
        $run_as_user = $this->entityTypeManager->getStorage('user')->load((int) $input);
      }
      else {
        // Load by username.
        $users = $this->entityTypeManager->getStorage('user')->loadByProperties(['name' => $input]);
        $run_as_user = reset($users);
      }

      if ($run_as_user) {
        $switcher->switchTo($run_as_user);
        $this->output()->writeln("<info>Running import as user: {$run_as_user->getAccountName()} (UID: {$run_as_user->id()})</info>");
      }
      else {
        $this->output()->writeln("<error>User '{$input}' not found. Falling back to current user.</error>");
      }
    }

    $entity_type = $options['entity-type'];
    if ($entity_type) {
      $this->output()->writeln("<info>Filtering import by entity type: {$entity_type}</info>");
    }

    $files = ExportDirectory::getFiles($entity_type ?: '*');

    $this->output()->writeln("<info>Starting import of " . count($files) . " files...</info>");
    $progress = new ProgressBar($this->output(), count($files));

    foreach ($files as $file_path) {
      $this->output()->writeln("\nProcessing file: $file_path");

      $json_string = file_get_contents($file_path);
      if ($json_string === FALSE) {
        $this->output()->writeln("<error>Failed to read file $file_path</error>");
        $progress->advance();
        continue;
      }

      $json_data = json_decode($json_string, TRUE);
      if ($json_data === NULL) {
        $this->output()->writeln("<error>Invalid JSON in $file_path</error>");
        $progress->advance();
        continue;
      }

      try {
        // Call importer; validation log is read from the importer static.
        $this->entityImporter->import($json_data);
        $log = $this->entityImporter::$entityValidation ?? [];

        foreach ($log as $l) {
          $this->output()->writeln("<comment>$l</comment>");
        }
        $this->output()->writeln("<info>Imported successfully.</info>");
        $this->entityImporter::$entityValidation = [];
      }
      catch (\Exception $e) {
        $this->output()->writeln("<error>Error importing $file_path: " . $e->getMessage() . "</error>");
      }

      $progress->advance();
    }

    $progress->finish();
    $this->output()->writeln("\n<info>All files processed.</info>");
  }

  /**
   * Lists the exported files by entity type.
   *
   * @command entity-io:list
   * @aliases eil
   * @option entity-type Entity type to filter by (e.g., node, taxonomy_term, user)
   *
   * @usage drush entity-io:list
   *   Lists all exported files.
   * @usage drush entity-io:list --entity-type=node
   *   Lists only the exported files for nodes.
   */
  public function listFiles(array $options = ['entity-type' => NULL]) {
    $entity_type = $options['entity-type'];
    $files = ExportDirectory::getFiles($entity_type ?: '*');

    if (empty($files)) {
      $msg = $entity_type
        ? "No files found for entity type '{$entity_type}'."
        : "No JSON file found.";
      $this->io()->warning($msg);
      return;
    }

    // Prepare data for display.
    $rows = [];
    foreach ($files as $file) {
      $entity_type_name = $this->extractEntityTypeFromPath($file);
      $file_name = basename($file);
      $real_path = $this->fileSystem->realpath($file);
      $file_size = file_exists($real_path) ? format_size(filesize($real_path)) : '-';

      $rows[] = [
        'entity_type' => $entity_type_name,
        'file_name' => $file_name,
        'file_path' => $file,
        'file_size' => $file_size,
      ];
    }

    // Sort by entity type and file name.
    usort($rows, fn($a, $b) => [$a['entity_type'], $a['file_name']] <=> [$b['entity_type'], $b['file_name']]);

    // Display the formatted table.
    $table = new Table($this->output());
    $table->setStyle('box');
    $table->setHeaders(['Entity Type', 'File Name', 'File Path', 'File Size']);
    foreach ($rows as $row) {
      $table->addRow([$row['entity_type'], $row['file_name'], $row['file_path'], $row['file_size']]);
    }

    $table->render();

    $this->io()->success(sprintf('Total files found: %d', count($rows)));
  }

  /**
   * Extracts the entity type from the file path.
   *
   * @param string $file_path
   *   The file path.
   *
   * @return string
   *   Detected entity type.
   */
  protected function extractEntityTypeFromPath(string $file_path): string {
    $parts = explode('/', str_replace('\\', '/', $file_path));
    return $parts[2] ?? 'unknown';
  }

  /**
   * Import JSON files from a folder with confirmation and progress bar.
   *
   * @param string $directory
   *   The directory containing JSON files.
   *
   * @command entity-io:import-folder
   * @aliases eioif
   *
   * @usage drush eioif /path/to/json
   * @usage drush eioif public://entity_io_exports
   */
  public function importFolder($directory) {
    // Validate directory.
    if (!is_dir($directory)) {
      $this->logger()->error(dt('The directory "@dir" does not exist.', ['@dir' => $directory]));
      return;
    }

    // Find JSON files.
    $files = glob($directory . '/*.json');
    if (empty($files)) {
      $this->logger()->warning('No JSON files found in the specified directory.');
      return;
    }

    // List files to user.
    $this->logger()->notice('The following JSON files were found:');
    foreach ($files as $file) {
      $this->output()->writeln(' - ' . basename($file));
    }

    // Ask for confirmation.
    $helper = $this->getHelper('question');
    $question = new ConfirmationQuestion(
      "\nAre you sure you want to import these files? (yes/no): ",
      FALSE
    );

    if (!$helper->ask($this->input(), $this->output(), $question)) {
      $this->logger()->notice('Import cancelled.');
      return;
    }

    // Start importing with progress bar.
    $this->output()->writeln("\nStarting import...\n");
    $progress = new ProgressBar($this->output(), count($files));
    $progress->start();

    foreach ($files as $file) {
      try {
        $json = file_get_contents($file);
        $data = json_decode($json, TRUE);

        if (json_last_error() !== JSON_ERROR_NONE) {
          $this->logger()->error(dt('Invalid JSON in file "@file".', ['@file' => basename($file)]));
          $progress->advance();
          continue;
        }

        // Use injected importer service.
        $this->entityImporter->import($data);

        $progress->advance();
      }
      catch (\Exception $e) {
        $this->logger()->error(dt('Failed to import file "@file": @message', [
          '@file' => basename($file),
          '@message' => $e->getMessage(),
        ]));
        $progress->advance();
      }
    }

    $progress->finish();
    $this->output()->writeln("\n<info>All imports processed.</info>");
  }

  /**
   * Clear all exported JSON files.
   *
   * @command entity-io:clear-exports
   * @aliases eioce
   *
   * @option entity-type Filter by entity type (node, user, taxonomy_term, etc). If not specified, all files are deleted.
   *
   * @usage drush entity-io:clear-exports
   * @usage drush entity-io:clear-exports --entity-type=node
   * @usage drush eioce --entity-type=user
   */
  public function clearExports(array $options = ['entity-type' => NULL]) {
    $entity_type = $options['entity-type'];

    if ($entity_type) {
      $this->output()->writeln("<info>Clearing exported files for entity type: {$entity_type}</info>");
    }
    else {
      $this->output()->writeln("<info>Clearing all exported files</info>");
    }

    $result = ExportDirectory::clearFiles($entity_type);

    if ($result['deleted_count'] > 0) {
      $this->output()->writeln("<info>Cleared {$result['deleted_count']} files</info>");

      if ($this->output()->isVerbose()) {
        $this->output()->writeln("<comment>Deleted files:</comment>");
        foreach ($result['deleted_files'] as $file) {
          $this->output()->writeln(" - {$file}");
        }
      }
    }
    else {
      if ($entity_type) {
        $this->output()->writeln("<comment>No files found for entity type '{$entity_type}' or directory does not exist.</comment>");
      }
      else {
        $this->output()->writeln("<comment>No files found to delete or directory does not exist.</comment>");
      }
    }
  }

  /**
   * Collect entities to export.
   */
  protected function collectEntities($all = FALSE, $entity_type = NULL, $entity_id = NULL) {
    // If specific ID is requested, load that entity.
    if ($entity_id && $entity_type) {
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
      if ($entity) {
        return [$entity];
      }
      else {
        $this->output()->writeln("<error>Entity {$entity_type}:{$entity_id} not found.</error>");
        return [];
      }
    }

    // If specific entity type is requested.
    if ($entity_type) {
      return $this->collectEntitiesByType($entity_type, $all);
    }

    // Default behavior - all entity types.
    if ($all) {
      return $this->collectEntitiesAll();
    }

    // Sample entities (1 per type)
    return $this->collectSampleEntities();
  }

  /**
   * Collect entities by specific type.
   */
  protected function collectEntitiesByType($entity_type, $all = FALSE) {
    $entities = [];

    try {
      $storage = $this->entityTypeManager->getStorage($entity_type);

      if ($all) {
        // Load all entities of this type.
        switch ($entity_type) {
          case 'taxonomy_term':
            foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocabulary) {
              $terms = $storage->loadByProperties(['vid' => $vocabulary->id()]);
              $entities = array_merge($entities, $terms);
            }
            break;

          case 'node':
            foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
              $nodes = $storage->loadByProperties(['type' => $type->id()]);
              $entities = array_merge($entities, $nodes);
            }
            break;

          case 'user':
            $users = $storage->loadMultiple();
            foreach ($users as $user) {
              // Exclude anonymous user.
              if ($user->id() > 0) {
                $entities[] = $user;
              }
            }
            break;

          case 'media':
            foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
              $media_items = $storage->loadByProperties(['bundle' => $type->id()]);
              $entities = array_merge($entities, $media_items);
            }
            break;

          default:
            $entities = $storage->loadMultiple();
            break;
        }
      }
      else {
        // Load sample entities (1 per bundle)
        switch ($entity_type) {
          case 'taxonomy_term':
            foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocabulary) {
              $terms = $storage->loadByProperties(['vid' => $vocabulary->id()]);
              if ($terms) {
                $entities[] = reset($terms);
              }
            }
            break;

          case 'node':
            foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
              $nodes = $storage->loadByProperties(['type' => $type->id()]);
              if ($nodes) {
                $entities[] = reset($nodes);
              }
            }
            break;

          case 'media':
            foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
              $media_items = $storage->loadByProperties(['bundle' => $type->id()]);
              if ($media_items) {
                $entities[] = reset($media_items);
              }
            }
            break;

          default:
            $all_entities = $storage->loadMultiple();
            if ($all_entities) {
              $entities[] = reset($all_entities);
            }
            break;
        }
      }
    }
    catch (\Exception $e) {
      $this->output()->writeln("<error>Error loading entities of type {$entity_type}: " . $e->getMessage() . "</error>");
    }

    return $entities;
  }

  /**
   * Collect sample entities (original behavior).
   */
  protected function collectSampleEntities() {
    $entities = [];

    // 1 term per vocabulary
    foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocabulary) {
      $terms = $this->entityTypeManager->getStorage('taxonomy_term')->loadByProperties(['vid' => $vocabulary->id()]);
      if ($terms) {
        $entities[] = reset($terms);
      }
    }

    // 1 node per content type
    foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
      $nodes = $this->entityTypeManager->getStorage('node')->loadByProperties(['type' => $type->id()]);
      if ($nodes) {
        $entities[] = reset($nodes);
      }
    }

    // 1 user
    $users = $this->entityTypeManager->getStorage('user')->loadMultiple();
    foreach ($users as $user) {
      if ($user->id() > 1) {
        $entities[] = $user;
        break;
      }
    }

    // 1 block
    $blocks = $this->entityTypeManager->getStorage('block_content')->loadMultiple();
    if ($blocks) {
      $entities[] = reset($blocks);
    }

    // 1 comment
    $comments = $this->entityTypeManager->getStorage('comment')->loadMultiple();
    if ($comments) {
      $entities[] = reset($comments);
    }

    // 1 media per type
    foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
      $media_items = $this->entityTypeManager->getStorage('media')->loadByProperties(['bundle' => $type->id()]);
      if ($media_items) {
        $entities[] = reset($media_items);
      }
    }

    return $entities;
  }

  /**
   * Collects all entities of the supported types.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   An array of entities.
   */
  protected function collectEntitiesAll() {
    $entities = [];

    // All taxonomy terms.
    foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocabulary) {
      $terms = $this->entityTypeManager
        ->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => $vocabulary->id()]);
      $entities = array_merge($entities, $terms);
    }

    // All nodes.
    foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
      $nodes = $this->entityTypeManager
        ->getStorage('node')
        ->loadByProperties(['type' => $type->id()]);
      $entities = array_merge($entities, $nodes);
    }

    // All users except anonymous.
    $users = $this->entityTypeManager->getStorage('user')->loadMultiple();
    foreach ($users as $user) {
      if ($user->id() > 0) {
        $entities[] = $user;
      }
    }

    // All blocks.
    $entities = array_merge($entities, $this->entityTypeManager->getStorage('block_content')->loadMultiple());

    // All comments.
    $entities = array_merge($entities, $this->entityTypeManager->getStorage('comment')->loadMultiple());

    // All media.
    foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
      $media_items = $this->entityTypeManager
        ->getStorage('media')
        ->loadByProperties(['bundle' => $type->id()]);
      $entities = array_merge($entities, $media_items);
    }

    return $entities;
  }

  /**
   * Export a single entity and save to JSON.
   */
  protected function saveExport($entity, array &$exported_files) {
    $max_depth = -1;
    $selected_fields = $this->entityIoExport->getSelectedFields($entity);
    $current_langcode = $entity->langcode->value;
    $exporter = $this->exporter;
    $exporter::toJson($entity, 0, $max_depth, $selected_fields, $current_langcode);

    if (PHP_SAPI === 'cli') {
      $exported_files[] = $exporter::$publicPathFile ?? ($exporter::$publicPathFile ?? '');
    }
    else {
      $exported_files[] = $exporter::$publicUrl ?? ($exporter::$publicUrl ?? '');
    }
  }

}
