<?php

namespace Drupal\entity_io\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller to handle file downloads from private storage.
 */
class FileDownloadController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs the controller.
   */
  public function __construct(ConfigFactoryInterface $config_factory, FileSystemInterface $file_system) {
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('file_system')
    );
  }

  /**
   * Download a file from storage.
   *
   * @param string $entity_type
   *   The entity type directory (ex: node, media).
   * @param string $filename
   *   The filename requested (ex: example.json).
   *
   * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
   *   The file download response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Thrown when file is not found.
   */
  public function download($entity_type, $filename) {
    $config = $this->configFactory->get('entity_io.export_settings');
    $scheme = $config->get('storage_scheme') ?? 'public';
    $directory = $config->get('directory') ?? 'entity_io_exports';

    // Build the stream wrapper path.
    $file_path = $scheme . '://' . $directory . '/' . $entity_type . '/' . $filename;

    // Resolve to real filesystem path using injected file system service.
    $real_path = $this->fileSystem->realpath($file_path);

    if (empty($real_path) || !file_exists($real_path)) {
      throw new NotFoundHttpException("File not found: $filename");
    }

    // Create a binary file response for download.
    $response = new BinaryFileResponse($real_path);
    $response->setContentDisposition(
      ResponseHeaderBag::DISPOSITION_ATTACHMENT,
      $filename
    );

    // Optionally set headers for security or cache control.
    $response->headers->set('Content-Type', mime_content_type($real_path) ?: 'application/octet-stream');
    $response->headers->set('Cache-Control', 'private');

    $response->send();
    die;
  }

}
