<?php

namespace Drupal\entity_io\Form\Export;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\entity_io\Service\EntityIoExport;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Generic form to export entity as JSON with selectable fields and AJAX output.
 */
class EntityIoExportGenericForm extends FormBase {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity_io.export service (concrete/instance).
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs the form.
   */
  public function __construct(LanguageManagerInterface $language_manager, EntityFieldManagerInterface $entity_field_manager, EntityIoExport $entity_io_export, $exporter, RequestStack $request_stack) {
    $this->languageManager = $language_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityIoExport = $entity_io_export;
    $this->exporter = $exporter;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('entity_field.manager'),
      $container->get('entity_io.export'),
      $container->get('entity_io.exporter'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_generic_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?EntityInterface $entity = NULL) {
    if (!$entity) {
      return ['#markup' => $this->t('Invalid entity.')];
    }

    $langcode = $this->languageManager->getCurrentLanguage()->getId();

    if ($entity->hasTranslation($langcode)) {
      $entity = $entity->getTranslation($langcode);
    }

    $entity_type_id = $entity->getEntityTypeId();

    // Use injected export service to obtain pre-selected fields.
    /** @var \Drupal\entity_io\Service\EntityIoExport $export_service */
    $export_service = $this->entityIoExport;
    $selected_fields = $export_service::getSelectedFields($entity);

    // Base fields via injected entity field manager.
    $base_fields = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
    $form['base_fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Base Fields'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];
    foreach ($base_fields as $field_name => $definition) {
      $default = in_array($field_name, $selected_fields, TRUE);
      $form['base_fields'][$field_name] = [
        '#type' => 'checkbox',
        '#title' => $field_name,
        '#default_value' => $default,
      ];
    }

    // Configurable fields.
    $form['fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Fields'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    foreach ($entity->getFields() as $field_name => $field) {
      if (str_starts_with($field_name, 'field_') || in_array($field_name, ['body', 'comment'])) {
        $default = in_array($field_name, $selected_fields, TRUE);
        $form['fields'][$field_name] = [
          '#type' => 'checkbox',
          '#title' => $field_name,
          '#default_value' => $default,
        ];
      }
    }

    // Depth selector.
    $options = [];
    for ($i = -1; $i <= 20; $i++) {
      $options[(string) $i] = (string) $i;
    }
    $form['depth'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity reference depth'),
      '#options' => $options,
      '#default_value' => -1,
    ];

    // Checkbox to pretty-print JSON.
    $form['pretty_print'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Format JSON output'),
      '#default_value' => FALSE,
      '#description' => $this->t('If checked, the exported JSON will be formatted for readability.'),
    ];

    // AJAX export button.
    $form['export'] = [
      '#type' => 'button',
      '#value' => $this->t('Export JSON'),
      '#ajax' => [
        'callback' => '::ajaxExport',
        'wrapper' => 'json-export-result',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating JSON...'),
        ],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['entity-io-export-btn']],
    ];

    // JSON result container.
    $form['result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'json-export-result'],
      'output' => [
        '#type' => 'textarea',
        '#title' => $this->t('Exported JSON'),
        '#value' => $form_state->get('exported_json') ?? '',
        '#rows' => 25,
        '#attributes' => [
          'readonly' => 'readonly',
          'style' => 'font-family: monospace; width: 100%;',
        ],
      ],
    ];

    // Add JS library.
    $form['#attached']['library'][] = 'entity_io/entity_io_export_scroll';

    return $form;
  }

  /**
   * AJAX callback for JSON export.
   */
  public function ajaxExport(array &$form, FormStateInterface $form_state) {
    $request = $this->requestStack->getCurrentRequest();
    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $request->attributes->get('node')
      ?? $request->attributes->get('taxonomy_term')
      ?? $request->attributes->get('block_content')
      ?? $request->attributes->get('user')
      ?? $request->attributes->get('comment')
      ?? $request->attributes->get('media');

    if (!$entity instanceof EntityInterface) {
      $form['result']['output']['#value'] = 'Invalid entity.';
      return $form['result'];
    }

    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $max_depth = (int) $form_state->getValue('depth');
    $pretty = (bool) $form_state->getValue('pretty_print');
    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $exporter = $this->exporter;
    $exporter::toJson($entity, 0, $max_depth, $selected_fields, $current_langcode);
    $json = $exporter::$json ?? '';

    // Pretty print if requested.
    if ($pretty) {
      $decoded = json_decode($json, TRUE);
      if (json_last_error() === JSON_ERROR_NONE) {
        $json = json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
      }
    }

    $form_state->set('exported_json', $json);
    $form['result']['output']['#value'] = $json;

    return $form['result'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Not used (AJAX only).
  }

}
