<?php

namespace Drupal\entity_io\Service;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\block_content\Entity\BlockContent;
use Drupal\comment\Entity\Comment;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\taxonomy\Entity\Term;
use Drupal\user\Entity\User;

/**
 * Exporting service EntityIoExport.
 */
final class EntityIoExport {

  /**
   * Current language code.
   *
   * @var string
   */
  protected static $langcode;

  /**
   * Debug mode.
   *
   * @var bool
   */
  protected static $debug;

  /**
   * A global static property to store exported entities.
   *
   * @var array
   */
  public static $exportedEntities = [];

  /**
   * Enable debug mode.
   */
  public static function setDebug($boolean) {
    self::$debug = TRUE;
  }

  /**
   * Export an entity to an array.
   */
  public static function export($entity, $depth, $max_depth, $selected_fields, $current_langcode) {
    self::$langcode = $current_langcode;
    if ($entity->hasTranslation($current_langcode)) {
      $entity = $entity->getTranslation($current_langcode);
    }
    $languages = $entity->getTranslationLanguages();
    if (isset($languages[self::$langcode])) {
      $data = self::exportEntity($entity, $depth, $max_depth, $selected_fields);
      $data['__entity_type__'] = $entity->getEntityTypeId();
    }
    if (isset($languages[$current_langcode])) {
      unset($languages[$current_langcode]);
    }

    foreach ($languages as $langcode => $language) {
      self::$langcode = $langcode;
      if ($entity->hasTranslation(self::$langcode)) {
        $entity = $entity->getTranslation($langcode);
        self::exportEntity($entity, $depth, $max_depth, $selected_fields);
        $data['translations'][] = $langcode;
      }
    }
    $data['exportedEntities'] = self::$exportedEntities;

    \Drupal::service('entity_io.logger')->log('export', $entity->getEntityTypeId(), $entity->id(), 'success', 'Export successful');

    return $data;
  }

  /**
   * Export a single entity.
   */
  public static function exportEntity($entity, $depth, $max_depth, $selected_fields) {
    $data = [];

    if ($entity->hasTranslation(self::$langcode)) {
      $entity = $entity->getTranslation(self::$langcode);

      self::$exportedEntities[$entity->uuid()][self::$langcode] = $entity->uuid();
      foreach ($entity->getFields() as $field) {
        if ($selected_fields && !in_array($field->getName(), $selected_fields)) {
          continue;
        }
        $data[$field->getName()] = self::getFieldValue($field, $depth + 1, $max_depth);
        if ($data[$field->getName()] === NULL) {
          unset($data[$field->getName()]);
        }
      }
    }
    else {
      foreach ($entity->getFields() as $field) {
        if ($selected_fields && !in_array($field->getName(), $selected_fields)) {
          continue;
        }
        $data[$field->getName()] = self::getFieldValue($field, $depth + 1, $max_depth);
        if ($data[$field->getName()] === NULL) {
          unset($data[$field->getName()]);
        }
      }
    }
    if ($data) {
      $data['__entity_type__'] = $entity->getEntityTypeId();
      self::$exportedEntities[$entity->uuid()][self::$langcode] = $data;
    }
    return $data;
  }

  /**
   * Auto-detect field type and print its value.
   */
  public static function getFieldValue(FieldItemListInterface $field, $depth = -1, $max_depth = -1) {
    $field_definition = $field->getFieldDefinition();
    $field_type = $field_definition->getType();
    $target_type = $field_definition->getSetting('target_type');
    $field_name = $field->getName();

    if ($depth > $max_depth && $max_depth != -1) {
      return $field->getValue();
    }

    switch ($field_type) {
      case 'uuid':
      case 'created':
      case 'changed':
      case 'language':
      case 'string':
      case 'password':
      case 'string_long':
        return self::getPlainText($field);

      case 'metatag_computed':
        return self::getArray($field);

      case 'metatag':
        return self::getPlainText($field);

      case 'map':
        return self::getArray($field);

      case 'text':
      case 'text_long':
      case 'text_with_summary':
        return self::printFormattedText($field);

      case 'integer':
      case 'decimal':
      case 'float':
        return self::printNumber($field);

      case 'list_string':
      case 'list_integer':
      case 'list_float':
        return self::getSelectionList($field);

      case 'datetime':
      case 'timestamp':
        return self::getDateTime($field);

      case 'boolean':
        return self::getBoolean($field);

      case 'email':
        return self::getEmail($field);

      case 'link':
        return self::getLink($field);

      case 'file_uri':
        return self::getFileUri($field);

      case 'image':
        return self::getImage($field, $depth + 1, $max_depth);

      case 'file':
        return self::getFile($field, $depth + 1, $max_depth);

      case 'comment':
        return self::getComment($field, $depth + 1, $max_depth);

      case 'uri':
        return $field->getValue();

      case 'daterange':
        return $field->getValue();

      case 'telephone':
        return $field->getValue();

      case 'path':
        return $field->getValue();

      case 'entity_reference':
      case 'entity_reference_revisions':
        $items = [];
        if ($target_type === 'paragraph') {
          $items = self::getParagraph($field, $depth + 1, $max_depth);
        }
        elseif ($target_type === 'paragraphs_type') {
          $items = self::getParagraphType($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'taxonomy_term') {
          $items = self::getTaxonomyTerm($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'taxonomy_vocabulary') {
          $items = self::getTaxonomyVocabulary($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'user') {
          $items = self::getUser($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'node_type') {
          $items = self::getNodeType($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'media') {
          $items = self::getMedia($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'media_type') {
          $items = self::getMediaType($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'node') {
          $items = self::getNode($field, $depth + 1, $max_depth);
        }
        elseif ($target_type == 'comment') {
          return self::getValue($field);
        }
        elseif ($target_type == 'user_role') {
          // @todo check this field role as entity exportable
          return self::getValue($field);
        }
        elseif ($target_type == 'node') {
          return self::getValue($field);
        }
        elseif ($target_type == 'comment_type') {
          return self::getValue($field);
        }
        elseif ($target_type == 'menu_link_content') {
          return self::getValue($field);
        }
        elseif ($target_type == 'block_content_type') {
          return self::getBlockContentType($field, $depth + 1, $max_depth);
        }
        else {
          \Drupal::service('entity_io.logger')->log('export', '', '', 'Export Failed', 'Unsupported target_type (' . $target_type . ') $field: ' . $field_name . ' $field_type: ' . $field_type);
          return NULL;
        }
        return $data[$field_name] = $items;

      default:
        \Drupal::service('entity_io.logger')->log('export', '', '', 'Export Failed', 'Unsupported field_type (' . $field_type . ') $field: ' . $field_name . ' $target_type: ' . $target_type);
        return NULL;
    }
  }

  /**
   * Generic get value function.
   */
  public static function getValue($field) {
    return $field->getValue();
  }

  /**
   * Export media entities.
   */
  public static function getMedia(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
        $items[] = [
          'target__uuid' => $referenced_entity->uuid(),
          'target__id' => $referenced_entity->id(),
          'langcode' => $referenced_entity->langcode->value,
          'already_exported' => TRUE,
          'log' => 'Already exported. Prevent infinite recursion',
          'token' => uniqid(),
        ];
        continue;
      }
      self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
      $selected_fields = self::getSelectedFields($referenced_entity);
      $items[] = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);
    }
    return $items;
  }

  /**
   * Export node entities.
   */
  public static function getNode(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
        $items[] = [
          'target__uuid' => $referenced_entity->uuid(),
          'target__id' => $referenced_entity->id(),
          'langcode' => $referenced_entity->langcode->value,
          'already_exported' => TRUE,
          'log' => 'Already exported. Prevent infinite recursion',
          'token' => uniqid(),
        ];
        continue;
      }
      self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
      $selected_fields = self::getSelectedFields($referenced_entity);
      $items[] = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);
    }
    return $items;
  }

  /**
   * Export user entities.
   */
  public static function getUser(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    if ($field->entity) {
      foreach ($field->referencedEntities() as $referenced_entity) {
        if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
          $items[] = [
            'target__uuid' => $referenced_entity->uuid(),
            'target__id' => $referenced_entity->id(),
            'langcode' => $referenced_entity->langcode->value,
            'already_exported' => TRUE,
            'log' => 'Already exported. Prevent infinite recursion',
            'token' => uniqid(),
          ];
          continue;
        }
        self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
        $selected_fields = self::getSelectedFields($referenced_entity);
        $items[] = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);
      }
    }
    return $items;
  }

  /**
   * Export taxonomy term entities.
   */
  public static function getTaxonomyTerm(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    if ($field->entity) {
      $items = [];
      foreach ($field->referencedEntities() as $referenced_entity) {
        if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
          $items[] = [
            'target__uuid' => $referenced_entity->uuid(),
            'target__id' => $referenced_entity->id(),
            'langcode' => $referenced_entity->langcode->value,
            'already_exported' => TRUE,
            'log' => 'Already exported. Prevent infinite recursion',
            'token' => uniqid(),
          ];
          continue;
        }
        self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
        $selected_fields = self::getSelectedFields($referenced_entity);
        $items[] = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);
      }
      return $items;
    }
    else {
      return NULL;
    }
  }

  /**
   * Export paragraph entities.
   */
  public static function getParagraph(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
        $items[] = [
          'target__uuid' => $referenced_entity->uuid(),
          'target__id' => $referenced_entity->id(),
          'langcode' => $referenced_entity->langcode->value,
          'already_exported' => TRUE,
          'log' => 'Already exported. Prevent infinite recursion',
          'token' => uniqid(),
        ];
        continue;
      }
      self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
      $selected_fields = self::getSelectedFields($referenced_entity);
      $items[] = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);
    }
    return $items;
  }

  /**
   * Export file entities.
   */
  public static function getFile($field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
        $items[] = [
          'target__uuid' => $referenced_entity->uuid(),
          'target__id' => $referenced_entity->id(),
          'langcode' => $referenced_entity->langcode->value,
          'already_exported' => TRUE,
          'log' => 'Already exported. Prevent infinite recursion',
          'token' => uniqid(),
        ];
        continue;
      }
      self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
      $selected_fields = self::getSelectedFields($referenced_entity);
      $data = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);

      $uri = $referenced_entity->getFileUri();
      $data['base64'] = '';
      if ($uri) {
        $real_path = \Drupal::service('file_system')->realpath($uri);
        $base64 = base64_encode(file_get_contents($real_path));
        $data['base64'] = $base64;
      }
      $items[] = $data;
    }
    return $items;
  }

  /**
   * Render an image field as <img>.
   */
  public static function getImage($field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $key => $referenced_entity) {
      if (self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] ?? FALSE) {
        $items[] = [
          'target__uuid' => $referenced_entity->uuid(),
          'target__id' => $referenced_entity->id(),
          'langcode' => $referenced_entity->langcode->value,
          'already_exported' => TRUE,
          'log' => 'Already exported. Prevent infinite recursion',
          'token' => uniqid(),
        ];
        continue;
      }
      self::$exportedEntities[$referenced_entity->uuid()][self::$langcode] = $referenced_entity->uuid();
      // @todo validate why $referenced_entity->bundle() return value file
      $selected_fields = self::getSelectedFields($referenced_entity);
      $data = self::exportEntity($referenced_entity, $depth, $max_depth, $selected_fields);

      if (self::$debug) {
        // dd($field, $referenced_entity, $field->getParent()->getEntity());
      }
      $data['alt'] = $field->getParent()->getEntity()->get($field->getName())[$key]->alt ?? '';
      $data['title'] = $field->getParent()->getEntity()->get($field->getName())[$key]->title ?? '';

      $uri = $referenced_entity->getFileUri();
      $data['base64'] = '';
      if ($uri) {
        $real_path = \Drupal::service('file_system')->realpath($uri);
        if ($real_path) {
          $base64 = base64_encode(file_get_contents($real_path));
          $data['base64'] = $base64;
        }
      }
      $items[] = $data;
    }
    return $items;
  }

  /**
   * Export comment entities.
   */
  public static function getComment($field, $depth, $max_depth) {
    $items = [];
    if ($field->getParent()->getEntity() instanceof Node) {
      $comment_storage = \Drupal::entityTypeManager()->getStorage('comment');
      $node = $field->getParent()->getEntity();
      $comments = $comment_storage->loadThread($node, $field->getName(), 'thread');

      foreach ($comments as $comment) {
        $items[] = self::getCommentWithReplies($comment->cid->value, $depth, $max_depth);
      }
    }
    return $items;
  }

  /**
   * Export comments.
   */
  public static function getCommentWithReplies($cid, $depth, $max_depth) {
    $comment = Comment::load($cid);

    if (!$comment instanceof Comment) {
      return [];
    }

    $selected_fields = self::getSelectedFields($comment);

    // Get comment fields.
    $fields = [];
    foreach ($comment->getFields() as $field_name => $field) {
      if (!in_array($field_name, $selected_fields)) {
        continue;
      }
      $field_name = $field->getName();
      $fields[$field_name] = self::getFieldValue($field, $depth, $max_depth);
    }
    $fields['__entity_type__'] = 'comment';

    // Find replies where 'pid' = $cid.
    $replies = \Drupal::entityTypeManager()
      ->getStorage('comment')
      ->loadByProperties(['pid' => $cid]);

    // Recursively export replies.
    $fields['replies'] = [];
    foreach ($replies as $reply) {
      $fields['replies'][] = self::getCommentWithReplies($reply->id(), $depth, $max_depth);
    }

    return $fields;
  }

  /**
   * Export node type entities.
   */
  public static function getNodeType(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      $items = [];
      $arr = [];
      $arr['id'] = [['value' => $referenced_entity->id()]];
      $arr['name'] = [['value' => $referenced_entity->label()]];
      $arr['uuid'] = [['value' => $referenced_entity->get('uuid')]];
      $arr['langcode'] = [['value' => $referenced_entity->get('langcode')]];
      $arr['originalId'] = [['value' => $referenced_entity->getOriginalId()]];
      $arr['description'] = [['value' => $referenced_entity->get('description')]];
      $arr['help'] = [['value' => $referenced_entity->get('help')]];
      $arr['new_revision'] = [['value' => $referenced_entity->get('new_revision')]];
      $arr['preview_mode'] = [['value' => $referenced_entity->get('preview_mode')]];
      $arr['display_submitted'] = [['value' => $referenced_entity->get('display_submitted')]];
      $items[] = $arr;
    }
    return $items;
  }

  /**
   * Export block type entities.
   */
  public static function getBlockContentType(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      $items = [];
      $arr = [];
      $arr['id'] = [['value' => $referenced_entity->id()]];
      $arr['name'] = [['value' => $referenced_entity->label()]];
      $arr['uuid'] = [['value' => $referenced_entity->get('uuid')]];
      $arr['langcode'] = [['value' => $referenced_entity->get('langcode')]];
      $arr['originalId'] = [['value' => $referenced_entity->getOriginalId()]];
      $arr['description'] = [['value' => $referenced_entity->get('description')]];
      $items[] = $arr;
    }
    return $items;
  }

  /**
   * Export media type entities.
   */
  public static function getMediaType(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      $arr = [];
      $arr['id'] = [['value' => $referenced_entity->id()]];
      $arr['name'] = [['value' => $referenced_entity->label()]];
      $arr['uuid'] = [['value' => $referenced_entity->get('uuid')]];
      $arr['langcode'] = [['value' => $referenced_entity->get('langcode')]];
      $arr['originalId'] = [['value' => $referenced_entity->getOriginalId()]];
      $arr['description'] = [['value' => $referenced_entity->get('description')]];
      $arr['source'] = [['value' => $referenced_entity->get('source')]];
      $items[] = $arr;
    }
    return $items;
  }

  /**
   * Export taxonomy vocabulary entities.
   */
  public static function getTaxonomyVocabulary(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      $arr = [];
      $arr['vid'] = [['value' => $referenced_entity->id()]];
      $arr['name'] = [['value' => $referenced_entity->label()]];
      $arr['uuid'] = [['value' => $referenced_entity->get('uuid')]];
      $arr['langcode'] = [['value' => $referenced_entity->get('langcode')]];
      $arr['originalId'] = [['value' => $referenced_entity->getOriginalId()]];
      $arr['description'] = [['value' => $referenced_entity->get('description')]];
      $items[] = $arr;
    }
    return $items;
  }

  /**
   * Export paragraph type entities.
   */
  public static function getParagraphType(FieldItemListInterface $field, $depth, $max_depth) {
    $items = [];
    foreach ($field->referencedEntities() as $referenced_entity) {
      $arr = [];
      $arr['id'] = [['value' => $referenced_entity->id()]];
      $arr['name'] = [['value' => $referenced_entity->label()]];
      $arr['uuid'] = [['value' => $referenced_entity->get('uuid')]];
      $arr['langcode'] = [['value' => $referenced_entity->get('langcode')]];
      $arr['originalId'] = [['value' => $referenced_entity->getOriginalId()]];
      $arr['icon_uuid'] = [['value' => $referenced_entity->get('icon_uuid')]];
      $arr['description'] = [['value' => $referenced_entity->get('description')]];
      $arr['icon_default'] = [['value' => $referenced_entity->get('icon_default')]];
      $items[] = $arr;
    }
    return $items;
  }

  /**
   * Array in a readable format.
   */
  public static function getArray($field) {
    return $field->getValue();
  }

  /**
   * Get File uri.
   */
  public static function getFileUri(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Plain text field.
   */
  public static function getPlainText(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Formatted text field.
   */
  public static function printFormattedText(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Number field.
   */
  public static function printNumber(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Selection list field (list_string, list_integer, list_float).
   */
  public static function getSelectionList(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Date and time field.
   */
  public static function getDateTime(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Boolean field.
   */
  public static function getBoolean(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Email field.
   */
  public static function getEmail(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      return $item->getValue();
      // Return $item->value;.
    });
  }

  /**
   * Link field.
   */
  public static function getLink(FieldItemListInterface $field) {
    return self::getMultipleValues($field, function ($item) {
      $value = $item->getValue();
      if (isset($value['uri']) && preg_match('/^entity:([a-z_]+)\/(\d+)$/', $value['uri'], $matches)) {
        $entity_type = $matches[1];
        $entity_id   = (int) $matches[2];

        $entity_storage = \Drupal::entityTypeManager()->getStorage($entity_type);
        $entity = $entity_storage->load($entity_id);
        if ($entity) {
          $value['target__uuid'] = $entity->uuid();
        }
      }
      return $value;
      // Return $item->value;.
    });
  }

  /**
   * Helper function to handle multiple or single values.
   */
  protected static function getMultipleValues(FieldItemListInterface $field, callable $callback) {
    $output = [];
    foreach ($field as $item) {
      $output[] = $callback($item);
    }
    return $output;
  }

  /**
   * Return selected fields by entity type.
   */
  public static function getSelectedFields($entity) {
    $fields_config = self::getConfigEntity($entity);

    if (empty($fields_config)) {
      return [];
    }

    if ($entity instanceof Node) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof User) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof Term) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof BlockContent) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof Paragraph) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof Media) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof Comment) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }
    if ($entity instanceof File) {
      $fields = array_merge(array_keys($fields_config['fields'] ?? []), array_keys($fields_config['base_fields'] ?? []));
    }

    if (empty($fields)) {
      return self::getAllFields($entity);
    }

    return $fields;
  }

  /**
   * Return all fields by entity type.
   */
  public static function getAllFields($entity) {
    $fields = [];
    $field_definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions($entity->getEntityTypeId(), $entity->bundle());

    foreach ($field_definitions as $field_name => $field_definition) {
      $fields[] = $field_name;
    }

    $fields_ = \Drupal::entityTypeManager()->getStorage('field_config')->loadByProperties([
      'entity_type' => $entity->getEntityTypeId(),
      'bundle' => $entity->bundle(),
    ]);

    if ($fields_) {
      foreach ($fields_ as $field) {
        $fields[] = $field->getName();
      }
    }
    return $fields;
  }

  /**
   * Return selected fields by entity type.
   */
  public static function getDepth($entity) {
    $fields_config = self::getConfigEntity($entity);
    return $fields_config['depth'] ?? -1;
  }

  /**
   * Return config entity by entity type.
   */
  public static function getConfigEntity($entity) {
    $fields_config = [];
    if ($entity instanceof Node) {
      $node_config = \Drupal::config('entity_io.content_type_field_settings');
      $fields_config = $node_config->get($entity->bundle());
    }
    if ($entity instanceof User) {
      $user_config = \Drupal::config('entity_io.user_field_settings');
      $fields_config = $user_config->get($entity->bundle());
    }
    if ($entity instanceof Term) {
      $taxonomy_term_config = \Drupal::config('entity_io.taxonomy_term_field_settings');
      $fields_config = $taxonomy_term_config->get($entity->bundle());
    }
    if ($entity instanceof BlockContent) {
      $block_config = \Drupal::config('entity_io.block_content_field_settings');
      $fields_config = $block_config->get($entity->bundle());
    }
    if ($entity instanceof Paragraph) {
      $paragraph_config = \Drupal::config('entity_io.paragraph_field_settings');
      $fields_config = $paragraph_config->get($entity->bundle());
    }
    if ($entity instanceof Media) {
      $media_config = \Drupal::config('entity_io.media_field_settings');
      $fields_config = $media_config->get($entity->bundle());
    }
    if ($entity instanceof Comment) {
      $comment_config = \Drupal::config('entity_io.comment_field_settings');
      $fields_config = $comment_config->get('comment');
    }
    if ($entity instanceof File) {
      $file_config = \Drupal::config('entity_io.file_field_settings');
      $fields_config = $file_config->get($entity->bundle());
    }

    return $fields_config;
  }

  /**
   * Recursively checks if an entity orts parent entities has the ID.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param int $target_id
   *   The ID to match against parent entities.
   *
   * @return bool
   *   TRUE if a parent entity with the target ID is found, FALSE otherwise.
   */
  public static function isRecursive(EntityInterface $entity, int $target_id): bool {
    if (!method_exists($entity, 'getParent')) {
      return FALSE;
    }

    $parent = $entity->getParent()->getEntity();

    if (!$parent) {
      return FALSE;
    }

    if ($parent->id() === $target_id) {
      return TRUE;
    }

    // Recursive call.
    return self::isRecursive($parent, $target_id);
  }

}
