<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\entity_io\Helper\EntityJsonDiff;
use Drupal\entity_io\Helper\EntityLoader;
use Drupal\entity_io\Helper\JsonParseEntityData;
use Drupal\entity_io\Helper\JsonValidate;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Form to import Drupal entities from a JSON file.
 *
 * This form allows administrators or developers to upload a JSON file
 * containing serialized entity data, validate the contents, preview
 * the differences between existing and new data, and optionally
 * create or overwrite entities in the system.
 */
class EntityJsonImportForm extends FormBase {

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The entity importer service.
   *
   * @var mixed
   */
  protected $entityImporter;

  /**
   * The entity IO storage service.
   *
   * @var mixed
   */
  protected $entityIoStorage;

  /**
   * Constructs the form.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param mixed $entity_importer
   *   The entity importer service (entity_io.entity_importer).
   * @param mixed $entity_io_storage
   *   The entity IO storage service (entity_io.storage).
   */
  public function __construct(RequestStack $request_stack, EntityTypeManagerInterface $entity_type_manager, RendererInterface $renderer, $entity_importer, $entity_io_storage) {
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
    $this->entityImporter = $entity_importer;
    $this->entityIoStorage = $entity_io_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('entity_io.entity_importer'),
      $container->get('entity_io.storage')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_entity_json_import_form';
  }

  /**
   * {@inheritdoc}
   *
   * Builds the multi-step form for uploading and validating
   * the JSON file.
   * Step 1: Upload JSON file.
   * Step 2: Preview and confirm import with diff visualization.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Entity JSON Import'),
      '#description' => $this->t('Upload a JSON file containing a Drupal entity. You will be able to preview the changes before confirming the import.'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    // Step 1: Upload JSON file if data is not yet loaded into the form state.
    $form['fieldset']['json_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Upload JSON file'),
      '#description' => $this->t('Upload a JSON file containing an entity to import.'),
      '#required' => !$form_state->get('json_data'),
    ];

    // Step 2: Display validation results and diff
    // preview if JSON data is already available.
    if ($json_data = $form_state->get('json_data')) {
      unset($form['json_file']);
      $json_data = $json_data[0];

      // Normalize JSON data to prevent language-related
      // issues during entity creation.
      $json_data = JsonParseEntityData::preventEntityCreationLanguage($json_data);
      // Validate the structure and fields of the uploaded JSON.
      $jsonValidation = JsonValidate::validateEntity($json_data);
      $jsonLogs = JsonValidate::getLog();
      $jsonErrors = JsonValidate::getErrors();

      // Stop processing if validation fails.
      if (!$jsonValidation) {
        // $this->messenger()->addError($this->t('Invalid JSON data.'));
        foreach ($jsonLogs as $log) {
          $this->messenger()->addError($log);
        }
        foreach ($jsonErrors as $error) {
          $this->messenger()->addError($error);
        }
        // Stop rendering preview step.
        $form_state->set('json_data', NULL);
        $form_state->setRebuild(TRUE);
        // Initial submit button for validating and continuing to preview step.
        $form['submit'] = [
          '#type' => 'submit',
          '#value' => $this->t('Validate & Continue'),
        ];

        return $form;
      }

      // Parse metadata such as entity type, bundle, UUID, and title.
      $entityData = JsonParseEntityData::parseEntityInfo($json_data);
      extract($entityData);

      // Ensure the JSON contains the required entity metadata.
      if (!$type || !$bundle || !$uuid) {
        $this->messenger()->addError($this->t('Invalid JSON entity. Type, bundle, and UUID are required.'));
        // Stop rendering preview step.
        $form_state->set('json_data', NULL);
        $form_state->setRebuild(TRUE);
        return $form;
      }

      // Attempt to load an existing entity by UUID.
      $entity = $this->entityTypeManager
        ->getStorage($json_data['__entity_type__'])
        ->loadByProperties(['uuid' => $uuid]);

      $entity_bind = FALSE;
      // If the entity is not found, check if a previously
      // imported binding exists.
      if (!$entity) {
        $bind = $this->entityIoStorage->get($uuid);
        if ($bind) {
          $entity_bind = TRUE;
          $entity = EntityLoader::loadEntity($bind);
        }
      }

      // Build form elements for preview and confirmation.
      if ($entity) {
        // Case where entity already exists in the system.
        $form['preview'] = [
          '#type' => 'details',
          '#title' => $this->t('Import Preview'),
          '#open' => TRUE,
          'uuid' => ['#markup' => '<strong>UUID:</strong> ' . ($uuid . '<br>' ?? '-')],
        ];

        // Display message if binding created a new entity reference.
        if ($entity_bind) {
          $form['preview']['uuid']['#markup'] = '<strong>UUID:</strong> Entity <b>' . $uuid . '</b> did not exist and a new entity <b><i>' . $entity->uuid() . '</b></i> was created';
          $form['preview']['container_description'] = [
            '#type' => 'fieldset',
            'description' => ['#markup' => "Some fields may appear as automatic changes, such as <b>'id', 'uid', 'fid', 'tid', 'cid', 'mid', 'nid', 'uuid', 'vid', 'changed', 'langcode', 'revision_id', 'revision_uid', 'content_translation_uid', 'content_translation_changed', 'revision_timestamp', 'default_langcode' </b>, among others."],
          ];
        }

        // Display entity metadata.
        $form['preview']['type'] = ['#markup' => '<strong>Entity type:</strong> ' . ($type . '<br>' ?? '-')];
        $form['preview']['bundle'] = ['#markup' => '<strong>Bundle:</strong> ' . $bundle . '<br>'];
        $form['preview']['title'] = ['#markup' => '<strong>Title:</strong> ' . $title . '<br>'];
        $form['preview']['available_languages'] = ['#markup' => '<strong>Languages:</strong> ' . implode(', ', $available_languages) . '<br>'];

        foreach ($available_languages as $l) {
          $json_data = JsonParseEntityData::preventEntityCreationLanguage($json_data, $l);
          // Generate a diff of the JSON data compared to the
          // current entity data.
          $diffValidation = EntityJsonDiff::diff($json_data);

          // Prepare a render array for the diff table.
          $diffBuilder = [
            '#theme' => 'diffs_table',
            '#diffs' => $diffValidation,
          ];

          // Render diff preview as HTML.
          $diffHtmlBuild = $this->renderer->renderPlain($diffBuilder);

          // Show differences for existing entities.
          $form['diff'][$l] = [
            '#type' => 'details',
            '#title' => $this->t('DIFF') . ' ' . strtoupper($l),
            '#open' => FALSE,
            'search' => [
              '#markup' => Markup::create('<div class="diffs-search-input"></div>'),
            ],

            // Add the JS via library.
            'compare' => [
              '#markup' => $diffHtmlBuild,
            ],
          ];

        }

        // Require confirmation for overwriting.
        $form['confirm'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('The entity may already exist. Do you want to overwrite/import anyway?'),
          '#required' => TRUE,
        ];

      }
      else {
        // Case where no entity exists and a new one will be created.
        $form['preview'] = [
          '#type' => 'details',
          '#title' => $this->t('Import Preview'),
          '#open' => TRUE,
          'new_entity' => ['#markup' => '<h2>It seems that this entity does not exist. So a new one will be created.</h2>'],
        ];

        foreach ($available_languages as $l) {
          $json_data = JsonParseEntityData::preventEntityCreationLanguage($json_data, $l);
          // Generate a diff of the JSON data compared to the
          // current entity data.
          $diffValidation = EntityJsonDiff::diff($json_data);

          // Prepare a render array for the diff table.
          $diffBuilder = [
            '#theme' => 'diffs_table',
            '#diffs' => $diffValidation,
          ];

          // Render diff preview as HTML.
          $diffHtmlBuild = $this->renderer->renderPlain($diffBuilder);

          // Show differences for existing entities.
          $form['diff'][$l] = [
            '#type' => 'details',
            '#title' => $this->t('DIFF') . ' ' . strtoupper($l),
            '#open' => FALSE,
            'search' => [
              '#markup' => Markup::create('<div class="diffs-search-input"></div>'),
            ],

            // Add the JS via library.
            'compare' => [
              '#markup' => $diffHtmlBuild,
            ],
          ];

        }

        $form['confirm'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('The entity does NOT exist. Do you want to create/import it anyway?'),
          '#required' => TRUE,
        ];
      }

      // Add action buttons for confirmation and cancel.
      $form['import'] = [
        '#type' => 'submit',
        '#value' => $this->t('Confirm and Import'),
      ];

      $form['cancel'] = [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#submit' => ['::cancelForm'],
      ];

      // Attach the diff rendering library for styling.
      $form['#attached'] = [
        'library' => [
          'entity_io/diff2html',
          'entity_io/diffs_search',
          'entity_io/toggle_diffs_entity_table',
        ],
      ];

      return $form;
    }

    // Initial submit button for validating and continuing to preview step.
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Validate & Continue'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * Validates the uploaded JSON file during the first step.
   * - Ensures the file is uploaded.
   * - Checks MIME type.
   * - Validates JSON format and decodes it.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (!$form_state->get('json_data')) {
      // Use RequestStack to retrieve uploaded files safely.
      $request = $this->requestStack->getCurrentRequest();
      $uploaded_files = $request->files->get('files') ?: [];
      $jsonUpload = $uploaded_files['json_file'] ?? NULL;
      if ($jsonUpload instanceof UploadedFile) {
        // Ensure uploaded file is a valid JSON file.
        $mime = $jsonUpload->getClientMimeType();
        if ($mime !== 'application/json' && $mime !== 'text/json' && $mime !== 'application/octet-stream') {
          $form_state->setErrorByName('json_file', $this->t('Please upload a valid .json file.'));
          return;
        }

        $dataString = file_get_contents($jsonUpload->getRealPath());
        $data = json_decode($dataString, TRUE);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
          $form_state->setErrorByName('json_file', $this->t('Invalid JSON format.'));
          return;
        }
      }
      else {
        $form_state->setErrorByName('json_file', $this->t('JSON File required!'));
        return;
      }

      // Save decoded JSON for the next step and rebuild the form.
      $form_state->set('json_data', $data);
      $form_state->setRebuild(TRUE);
    }
  }

  /**
   * {@inheritdoc}
   *
   * Handles submission for final import step.
   * Imports the entity/entities from JSON and redirects to the edit form
   * if only one entity was imported successfully.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $json_data = $form_state->get('json_data');
    if ($json_data) {
      // Use injected importer service instead of \Drupal::service().
      $entities = $this->entityImporter->import($json_data);
      $entities = array_values($entities);

      // Access the importer's static validation log via its class.
      $importer_class = get_class($this->entityImporter);
      $log = $importer_class::$entityValidation ?? [];
      foreach ($log as $l) {
        $this->messenger()->addWarning($l);
      }

      // Clear static validation log if present.
      if (property_exists($importer_class, 'entityValidation')) {
        $importer_class::$entityValidation = [];
      }

      if ($entities && count($entities) === 1 && $entities[0] instanceof EntityInterface) {
        $this->messenger()->addStatus($this->t('Entity imported successfully.'));
        $form_state->setRedirectUrl($entities[0]->toUrl('edit-form'));
        return;
      }

      $this->messenger()->addStatus($this->t('Entity import complete.'));
    }
    else {
      $this->messenger()->addError($this->t('No JSON data found to import.'));
    }
  }

  /**
   * Cancels the import process and resets the form state.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state object.
   */
  public function cancelForm(array &$form, FormStateInterface $form_state) {
    $form_state->set('json_data', NULL);
    $this->messenger()->addWarning($this->t('Import cancelled.'));
    $form_state->setRedirect('<front>');
  }

}
