<?php

namespace Drupal\entity_io_queue\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Entity IO queue management.
 */
class EntityIoQueueController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected QueueFactory $queueFactory;

  /**
   * The queue worker manager.
   *
   * @var \Drupal\Core\Queue\QueueWorkerManagerInterface
   */
  protected QueueWorkerManagerInterface $queueWorkerManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a EntityIoQueueController object.
   *
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue factory.
   * @param \Drupal\Core\Queue\QueueWorkerManagerInterface $queueWorkerManager
   *   The queue worker manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(QueueFactory $queueFactory, QueueWorkerManagerInterface $queueWorkerManager, Connection $database, DateFormatterInterface $date_formatter, LoggerChannelFactoryInterface $logger_factory) {
    $this->queueFactory = $queueFactory;
    $this->queueWorkerManager = $queueWorkerManager;
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('queue'),
      $container->get('plugin.manager.queue_worker'),
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('logger.factory')
    );
  }

  /**
   * Display all items in the export queue with execution options.
   *
   * Creates a table listing all queued export items with their metadata
   * and provides execute buttons for manual processing.
   */
  public function listExportItems() {
    $header = [
      'item_id' => $this->t('ID'),
      'entity_type' => $this->t('Entity Type'),
      'entity_id' => $this->t('Entity ID'),
      'langcode' => $this->t('Language'),
      'created' => $this->t('Created'),
      'actions' => $this->t('Actions'),
    ];

    $query = $this->database->select('queue', 'q')
      ->fields('q', ['item_id', 'data', 'created'])
      ->condition('name', 'entity_io_queue_export')
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(20);

    $result = $query->execute();
    $rows = [];

    foreach ($result as $row) {
      $data = @unserialize($row->data, ['allowed_classes' => FALSE]);
      $rows[] = [
        'item_id' => $row->item_id,
        'entity_type' => $data['entity_type'] ?? 'N/A',
        'entity_id' => $data['entity_id'] ?? 'N/A',
        'langcode' => $data['langcode'] ?? 'N/A',
        'created' => $this->dateFormatter->format($row->created, 'short'),
        'actions' => [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('Execute'),
            '#url' => Url::fromRoute('entity_io_queue.execute_item', ['item_id' => $row->item_id]),
            '#attributes' => ['class' => ['button']],
          ],
        ],
      ];
    }

    $build = [];
    $build['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Export Queue Items'),
    ];

    $build['fieldset']['process_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Process all'),
      '#url' => Url::fromRoute('entity_io_queue.process_all_export'),
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    if (!empty($rows)) {
      $build['fieldset']['count_items'] = [
        '#markup' => $this->t('Entity IO Export Queue (@count items) <br>', ['@count' => count($rows)]),
      ];
    }

    $build['fieldset']['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No items in export queue.'),
    ];

    $build['fieldset']['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Execute a specific export queue item by ID.
   *
   * @param int $item_id
   *   The queue item ID to execute.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response back to the queue list.
   */
  public function executeItem($item_id) {
    return $this->executeQueueItem('entity_io_queue_export', 'entity_io_queue_export', $item_id, 'entity_io_queue.items_export');
  }

  /**
   * Execute a specific import queue item by ID.
   *
   * @param int $item_id
   *   The queue item ID to execute.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response back to the queue list.
   */
  public function executeImportItem($item_id) {
    return $this->executeQueueItem('entity_io_queue_import', 'entity_io_queue_import', $item_id, 'entity_io_queue.items');
  }

  /**
   * Generic method to execute any queue item with error handling.
   *
   * Processes a queue item using its corresponding worker plugin,
   * handles success/failure states, and provides user feedback.
   *
   * @param string $queue_name
   *   The name of the queue.
   * @param string $worker_name
   *   The queue worker plugin ID.
   * @param int $item_id
   *   The queue item ID to execute.
   * @param string $redirect_route
   *   The route to redirect to after execution.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to the specified route.
   */
  protected function executeQueueItem($queue_name, $worker_name, $item_id, $redirect_route) {
    $queue = $this->queueFactory->get($queue_name);
    $worker = $this->queueWorkerManager->createInstance($worker_name);

    $items = $this->getQueueItems($queue, $queue_name);
    $target_item = NULL;

    foreach ($items as $item) {
      if ($item->item_id == $item_id) {
        $target_item = $item;
        break;
      }
    }

    if (!$target_item) {
      $this->messenger()->addError($this->t('Queue item not found.'));
      return new RedirectResponse(Url::fromRoute($redirect_route)->toString());
    }

    try {
      $worker->processItem($target_item->data);
      $queue->deleteItem($target_item);
      $this->messenger()->addStatus($this->t('Queue item executed successfully.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to execute queue item: @message', ['@message' => $e->getMessage()]));
    }

    return new RedirectResponse(Url::fromRoute($redirect_route)->toString());
  }

  /**
   * Retrieve queue items from database.
   *
   * @param \Drupal\Core\Queue\QueueInterface $queue
   *   The queue object (unused but kept for consistency).
   * @param string $queue_name
   *   The queue name to query.
   *
   * @return array
   *   Array of queue item objects with unserialized data.
   */
  protected function getQueueItems($queue, $queue_name) {
    $items = [];
    $connection = $this->database;

    $result = $connection->select('queue', 'q')
      ->fields('q')
      ->condition('name', $queue_name)
      ->execute();

    foreach ($result as $record) {
      // Unserialize the stored data for processing.
      // Use allowed_classes = FALSE to avoid object
      // instantiation vulnerabilities.
      $record->data = @unserialize($record->data, ['allowed_classes' => FALSE]);
      $items[] = $record;
    }

    return $items;
  }

  /**
   * Display paginated list of import queue items with execution options.
   *
   * @return array
   *   Render array containing table and pager elements.
   */
  public function list() {
    $header = [
      'item_id' => $this->t('Item ID'),
      'created' => $this->t('Created'),
      'actions' => $this->t('Actions'),
    ];

    $query = $this->database->select('queue', 'q')
      ->fields('q', ['item_id', 'data', 'created'])
      ->condition('name', 'entity_io_queue_import')
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(20);

    $result = $query->execute();
    $rows = [];

    foreach ($result as $row) {
      $rows[] = [
        'item_id' => $row->item_id,
        'created' => $this->dateFormatter->format($row->created, 'short'),
        'actions' => [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('Execute'),
            '#url' => Url::fromRoute('entity_io_queue.execute_import_item', ['item_id' => $row->item_id]),
            '#attributes' => ['class' => ['button']],
          ],
        ],
      ];
    }

    $build = [];

    $build['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Import Queue Items'),
    ];
    // Add a "Process all" action for the import queue.
    $build['fieldset']['process_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Process all'),
      '#url' => Url::fromRoute('entity_io_queue.process_all_import'),
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    if (!empty($rows)) {
      $build['fieldset']['count_items'] = [
        '#markup' => $this->t('Entity IO Import Queue (@count items)', ['@count' => count($rows)]),
      ];
    }

    $build['fieldset']['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No queue items found.'),
    ];

    $build['fieldset']['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Add item to import queue via JSON API endpoint.
   *
   * Accepts JSON payload and creates a new import queue item.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request containing JSON data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with success or error status.
   */
  public function addItem(Request $request) {
    // Parse JSON payload from request body.
    $data = json_decode($request->getContent(), TRUE);

    if (empty($data)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Invalid JSON payload',
      ], 400);
    }

    $queue = $this->queueFactory->get('entity_io_queue_import');
    $queue->createItem($data['data']);

    return new JsonResponse([
      'status' => 'success',
      'message' => 'Item added to the queue',
    ]);
  }

  /**
   * Add entity export data to the export queue.
   *
   * Creates a new export queue item with the provided entity data
   * and export configuration parameters.
   *
   * @param array $data
   *   Array containing entity type, ID, and export options.
   *
   * @return bool
   *   TRUE if item was successfully added, FALSE otherwise.
   */
  public function addExportItem(array $data) {
    try {
      $queue = $this->queueFactory->get('entity_io_queue_export');
      $item_id = $queue->createItem($data);

      if ($item_id) {
        $this->loggerFactory->get('entity_io_queue')->info('Export item added to queue: @type:@id', [
          '@type' => $data['entity_type'],
          '@id' => $data['entity_id'],
        ]);
        return TRUE;
      }

      return FALSE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('entity_io_queue')->error('Failed to add export item to queue: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Start processing all items in the export queue via batch.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Batch process response.
   */
  public function processAllExport() {
    return $this->processAllQueue('entity_io_queue_export', 'entity_io_queue_export', 'entity_io_queue.items_export');
  }

  /**
   * Start processing all items in the import queue via batch.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Batch process response.
   */
  public function processAllImport() {
    return $this->processAllQueue('entity_io_queue_import', 'entity_io_queue_import', 'entity_io_queue.items');
  }

  /**
   * Generic method to build and start a batch to process all items of a queue.
   *
   * @param string $queue_name
   *   The queue name.
   * @param string $worker_name
   *   The worker plugin id to use for processing.
   * @param string $redirect_route
   *   Route to redirect to after batch finishes.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Batch process response.
   */
  protected function processAllQueue($queue_name, $worker_name, $redirect_route) {
    $queue = $this->queueFactory->get($queue_name);
    $items = $this->getQueueItems($queue, $queue_name);

    if (empty($items)) {
      $this->messenger()->addStatus($this->t('No items to process.'));
      return new RedirectResponse(Url::fromRoute($redirect_route)->toString());
    }

    $operations = [];
    foreach ($items as $item) {
      // Each operation will call the static batchProcessItem method.
      $operations[] = [
        ['\Drupal\entity_io_queue\Controller\EntityIoQueueController', 'batchProcessItem'],
        [$item->item_id, $queue_name, $worker_name],
      ];
    }

    $batch = [
      'title' => $this->t('Processing queue: @queue', ['@queue' => $queue_name]),
      'operations' => $operations,
      'finished' => ['\Drupal\entity_io_queue\Controller\EntityIoQueueController::batchFinished', []],
      'init_message' => $this->t('Starting processing...'),
      'progress_message' => $this->t('Processed @current out of @total.'),
      'error_message' => $this->t('An error occurred during processing.'),
    ];

    // Set the batch and start processing, redirecting
    // back to the listing on finish.
    batch_set($batch);
    return batch_process(Url::fromRoute($redirect_route)->toString());
  }

  /**
   * Batch operation: process a single queue item.
   *
   * Static so it can be invoked by the batch runner.
   *
   * @param int $item_id
   *   The queue item id.
   * @param string $queue_name
   *   The queue name.
   * @param string $worker_name
   *   The queue worker plugin id.
   * @param array $context
   *   Batch context.
   */
  public static function batchProcessItem($item_id, $queue_name, $worker_name, array &$context) {
    // Get services via \Drupal::service since batch
    // runner recreates environment.
    $queue_factory = \Drupal::service('queue');
    $queue = $queue_factory->get($queue_name);
    $worker_manager = \Drupal::service('plugin.manager.queue_worker');
    $worker = $worker_manager->createInstance($worker_name);
    $connection = \Drupal::database();

    // Load the specific item by id.
    $record = $connection->select('queue', 'q')
      ->fields('q')
      ->condition('item_id', $item_id)
      ->execute()
      ->fetchObject();

    if (!$record) {
      // Nothing to do.
      return;
    }

    $record->data = @unserialize($record->data, ['allowed_classes' => FALSE]);

    try {
      $worker->processItem($record->data);
      // Delete the processed item from the queue.
      $queue->deleteItem($record);
      if (!isset($context['results']['processed'])) {
        $context['results']['processed'] = 0;
      }
      $context['results']['processed']++;
    }
    catch (\Exception $e) {
      // Log and add to context for reporting.
      \Drupal::logger('entity_io_queue')->error('Batch processing failed for item @id: @message', [
        '@id' => $item_id,
        '@message' => $e->getMessage(),
      ]);
      if (!isset($context['results']['errors'])) {
        $context['results']['errors'] = [];
      }
      $context['results']['errors'][] = $this->t('Item @id: @msg', ['@id' => $item_id, '@msg' => $e->getMessage()]);
    }
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch finished successfully.
   * @param array $results
   *   Results accumulated in $context['results'].
   * @param array $operations
   *   The operations executed.
   */
  public static function batchFinished($success, $results, $operations) {
    if ($success && !empty($results['processed'])) {
      \Drupal::messenger()->addStatus(\Drupal::translation()->translate('Processed @count items.', ['@count' => $results['processed']]));
    }
    elseif (!empty($results['errors'])) {
      foreach ($results['errors'] as $error) {
        \Drupal::messenger()->addError($error);
      }
    }
    else {
      \Drupal::messenger()->addStatus(\Drupal::translation()->translate('Batch processing finished.'));
    }
  }

}
