<?php

namespace Drupal\entity_io\Form\Export;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\comment\Entity\Comment;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Comment JSON export form.
 */
class CommentJsonExportForm extends FormBase {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The class resolver service.
   *
   * @var \Drupal\Core\DependencyInjection\ClassResolverInterface
   */
  protected $classResolver;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * Constructs the form.
   */
  public function __construct(LanguageManagerInterface $language_manager, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, ClassResolverInterface $class_resolver, $exporter) {
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->classResolver = $class_resolver;
    $this->exporter = $exporter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('class_resolver'),
      $container->get('entity_io.exporter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_comment_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $comment = NULL) {
    $language = $this->languageManager->getCurrentLanguage();
    if (!$comment instanceof Comment) {
      return ['error' => ['#markup' => 'Invalid comment']];
    }

    if ($comment->hasTranslation($language->getId())) {
      $comment = $comment->getTranslation($language->getId());
    }

    $config = $this->configFactory->get('entity_io.comment_field_settings');

    $entity_type = $this->entityTypeManager->getDefinition('comment');
    $base_fields = Comment::baseFieldDefinitions($entity_type);

    $form[$comment->bundle()]['base_fields'] = [
      '#type' => 'details',
      '#title' => 'Base Fields',
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      $default = $config->get($comment->bundle())['base_fields'][$base_field_name] ?? '';
      $form[$comment->bundle()]['base_fields'][$base_field_name] = [
        '#type' => 'checkbox',
        '#title' => $base_field_name ,
        '#default_value' => $default,
      ];
    }

    $form['fields'] = [
      '#type' => 'details',
      '#title' => 'Fields',
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    foreach ($comment->getFields() as $field_name => $field) {
      if (str_starts_with($field_name, 'field_')) {
        $default = $config->get($comment->bundle())['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->getName(),
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => 'Check All',
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => 'Clear All',
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => 'Toggle',
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $options = [];
    for ($i = -1; $i <= 100; $i++) {
      $options[(string) $i] = (string) $i;
    }
    // Depth level.
    $form['depth'] = [
      '#type' => 'select',
      '#title' => 'Entity reference depth',
      '#options' => $options,
      '#default_value' => $config->get($comment->bundle())['depth'] ?? -1,
    ];

    // Export button.
    $form['export'] = [
      '#type' => 'submit',
      '#value' => 'Export as JSON',
    ];

    $form['queue_export'] = [
      '#type' => 'button',
      '#value' => $this->t('Add to Export Queue'),
      '#ajax' => [
        'callback' => '::ajaxQueueExport',
        'wrapper' => 'queue-export-messages',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding to queue...'),
        ],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['queue-export-btn'],
      ],
    ];

    $form['queue_messages'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $comment = $this->getRequest()->attributes->get('comment');
    $max_depth = (int) $form_state->getValue('depth');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    if ($comment instanceof Comment) {
      $exporter = $this->exporter;
      $exporter::toJson($comment, 0, $max_depth, $selected_fields, $current_langcode);
      $json = $exporter::$json ?? json_encode([]);
      $fileName = $exporter::$fileName ?? 'export.json';

      $response = new Response($json);
      $response->headers->set('Content-Type', 'application/json');
      $response->headers->set('Content-Disposition', 'attachment; filename="' . $fileName . '"');
      $response->send();
      die;
    }
  }

  /**
   * AJAX callback for queue export button.
   */
  public function ajaxQueueExport(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields_bundle = $form_state->getValue($this->getRequest()->attributes->get('comment')->bundle())['base_fields'] ?? [];
    $base_fields = array_filter($base_fields_bundle);
    $max_depth = (int) $form_state->getValue('depth');
    $comment = $this->getRequest()->attributes->get('comment');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    $response = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    if ($comment instanceof Comment) {
      try {
        $data = [
          'entity_type' => $comment->getEntityTypeId(),
          'entity_id' => $comment->id(),
          'depth' => 0,
          'max_depth' => $max_depth,
          'selected_fields' => $selected_fields,
          'langcode' => $current_langcode,
          'is_revision' => FALSE,
        ];

        $controller = $this->classResolver->getInstanceFromDefinition('Drupal\entity_io_queue\Controller\EntityIoQueueController');
        $result = $controller->addExportItem($data);

        if ($result) {
          $response['message'] = [
            '#markup' => '<div class="messages messages--status">' .
            $this->t('Comment @title has been added to the export queue.', [
              '@title' => $comment->label(),
            ]) . '</div>',
          ];
        }
        else {
          $response['message'] = [
            '#markup' => '<div class="messages messages--error">' .
            $this->t('Failed to add comment to queue.') . '</div>',
          ];
        }
      }
      catch (\Exception $e) {
        $response['message'] = [
          '#markup' => '<div class="messages messages--error">' .
          $this->t('Failed to add comment to queue: @message', [
            '@message' => $e->getMessage(),
          ]) . '</div>',
        ];
      }
    }
    else {
      $response['message'] = [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('Invalid comment entity.') . '</div>',
      ];
    }

    return $response;
  }

}
