<?php

namespace Drupal\entity_io_push\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Node JSON Push servers.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['entity_io_push.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_io_push_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('entity_io_push.settings');
    $servers = $form_state->get('servers');

    // Load from config if first build.
    if ($servers === NULL) {
      $servers = $config->get('servers') ?? [];
      $form_state->set('servers', $servers);
    }

    $form['servers'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Server Name'),
        $this->t('Domain'),
        $this->t('Headers'),
        $this->t('User'),
        $this->t('Password'),
        $this->t('Remove'),
      ],
      '#empty' => $this->t('No servers configured yet.'),
    ];

    foreach ($servers as $id => $server) {
      $form['servers'][$id]['name'] = [
        '#type' => 'textfield',
        '#description' => $this->t('A unique name for this server.'),
        '#default_value' => $server['name'] ?? '',
        '#required' => TRUE,
      ];
      $form['servers'][$id]['url'] = [
        '#type' => 'url',
        '#description' => $this->t('Base URL of the remote Drupal site, e.g. https://example.com'),
        '#default_value' => $server['url'] ?? '',
        '#required' => TRUE,
      ];
      $form['servers'][$id]['headers'] = [
        '#type' => 'textarea',
        '#description' => $this->t('Headers to include in the request'),
        '#default_value' => $server['headers'] ?? '',
        '#required' => FALSE,
      ];
      $form['servers'][$id]['user'] = [
        '#type' => 'textfield',
        '#description' => $this->t('Username for HTTP Basic Auth'),
        '#default_value' => $server['user'] ?? '',
        '#required' => FALSE,
      ];
      $form['servers'][$id]['password'] = [
        '#type' => 'textfield',
        '#description' => $this->t('Password for HTTP Basic Auth'),
        '#default_value' => $server['password'] ?? '',
        '#required' => FALSE,
      ];
      $form['servers'][$id]['remove'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Remove'),
      ];
    }

    $form['add_server'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Server'),
      '#submit' => ['::addServer'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::ajaxRebuild',
        'wrapper' => 'entity-io-push-settings-form',
      ],
    ];

    $form['#prefix'] = '<div id="entity-io-push-settings-form">';
    $form['#suffix'] = '</div>';

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for rebuild.
   */
  public function ajaxRebuild(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * Custom submit handler to add a new server row dynamically.
   */
  public function addServer(array &$form, FormStateInterface $form_state): void {
    $servers = $form_state->get('servers') ?? [];
    $servers[] = [
      'name' => '',
      'url' => '',
      'headers' => '',
      'user' => '',
      'password' => '',
    ];
    $form_state->set('servers', $servers);
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $servers = $form_state->getValue('servers') ?? [];
    // Filter out removed servers.
    $servers = array_filter($servers, fn($server) => empty($server['remove']));

    $this->config('entity_io_push.settings')
      ->set('servers', $servers)
      ->save();

    $form_state->set('servers', $servers);

    parent::submitForm($form, $form_state);
  }

}
