<?php

namespace Drupal\entity_io_webhooks;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service to handle entity IO webhooks.
 */
class EntityIoWebHooks {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an EntityIoWebHooks object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Dispara os webhooks configurados para a entidade e evento especificados.
   */
  public static function trigger($entity, $event) {

    $config = \Drupal::config('entity_io_webhooks.settings');
    $entity_type_id = $entity->getEntityTypeId();
    $bundle_id = $entity->bundle();
    $base = "entities.$entity_type_id.$bundle_id";
    $bundle_config = $config->get($base) ?? [];

    // Check if current event is allowed.
    if (!empty($bundle_config['events']) && is_array($bundle_config['events'])) {
      if (is_array($bundle_config['events']) && !in_array($event, $bundle_config['events'])) {
        return;
      }
    }

    $selected_fields = \Drupal::service('entity_io.export')::getSelectedFields($entity);
    $exporter = \Drupal::service('entity_io.exporter');
    $exporter::toJson($entity, 0, -1, $selected_fields, $entity->langcode->value);
    $json = $exporter::$json;

    // Get json path.
    $publicRealPathFile = $exporter::$publicRealPathFile;

    // --- Webhook ---
    if (!empty($bundle_config['webhooks']['enabled']) && $bundle_config['webhooks']['enabled']) {
      \Drupal::service('entity_io_webhooks.manager')->trigger($entity, $json, $event);
    }

    // --- E-mail ---
    if (!empty($bundle_config['mail']['enabled']) && $bundle_config['mail']['enabled']) {
      $to = $bundle_config['mail']['email'] ?? NULL;
      $entity_type = $entity->getEntityTypeId();

      // Templates from config.
      $body_template = $bundle_config['mail']['body'] ?? '';
      $subject_template = $bundle_config['mail']['subject'] ?? '';

      // First, perform simple placeholder replacements that users may include:
      $simple_replacements = [
        '[entity:type]' => $entity_type,
        '[entity:id]' => method_exists($entity, 'id') ? $entity->id() : '',
        '[entity:title]' => method_exists($entity, 'label') ? $entity->label() : '',
      ];
      $body_template = strtr($body_template, $simple_replacements);
      $subject_template = strtr($subject_template, $simple_replacements);

      // Then run token replacement for any
      // remaining tokens, using 'entity' context.
      $token_service = \Drupal::token();
      $token_context = ['entity' => $entity];
      $token_options = [
        'langcode' => $entity->language()->getId(),
        'clear' => FALSE,
        'sanitize' => FALSE,
      ];
      $body = $token_service->replace($body_template, $token_context, $token_options);
      $subject = $token_service->replace($subject_template, $token_context, $token_options);

      \Drupal::service('entity_io_webhooks.mail')->trigger($entity, $publicRealPathFile, $event, $to, $subject, $body);
    }

    // --- FTP JSON ---
    if (!empty($bundle_config['ftp']['enabled']) && $bundle_config['ftp']['enabled']) {
      \Drupal::service('entity_io_webhooks.ftp')->trigger($entity, $publicRealPathFile, ['foo' => 'bar']);
    }

  }

}
