<?php

namespace Drupal\entity_io_webhooks\Form;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure webhook settings per entity bundle.
 *
 * Provides a form to configure webhook, email and FTP options per bundle.
 */
class EntityWebhookBundleForm extends ConfigFormBase {

  /**
   * Bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $bundleInfo;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Moderation information service (optional).
   *
   * May be NULL when content_moderation is not available.
   *
   * @var mixed|null
   */
  protected $moderationInfo;

  /**
   * Constructs the form object.
   */
  public function __construct(EntityTypeBundleInfoInterface $bundle_info, ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory, $moderation_info = NULL) {
    $this->bundleInfo = $bundle_info;
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->moderationInfo = $moderation_info;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $moderation = NULL;
    if ($container->has('content_moderation.moderation_information')) {
      $moderation = $container->get('content_moderation.moderation_information');
    }

    return new static(
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory'),
      $moderation
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['entity_io_webhooks.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_webhooks_bundle_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type_id = NULL) {
    $config = $this->config('entity_io_webhooks.settings');
    $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);

    if (empty($bundles)) {
      $bundles = [
        $entity_type_id => [
          'label' => $this->t('@entity', ['@entity' => $entity_type_id]),
        ],
      ];
    }

    $form['entity_type_id'] = [
      '#type' => 'value',
      '#value' => $entity_type_id,
    ];

    // Consider both modules: workflows + content_moderation (content_moderation
    // provides the moderation information service).
    $has_workflows = $this->moduleHandler->moduleExists('workflows')
      && $this->moduleHandler->moduleExists('content_moderation');

    // Prepare services we will use if workflows are enabled.
    $moderation_info = $has_workflows ? $this->moderationInfo : NULL;
    $entity_storage = $has_workflows ? $this->entityTypeManager->getStorage($entity_type_id) : NULL;

    foreach ($bundles as $bundle_id => $bundle) {
      $bundle_config = $config->get("entities.$entity_type_id.$bundle_id") ?? [];

      $form[$bundle_id] = [
        '#type' => 'details',
        '#title' => $bundle['label'],
        '#open' => FALSE,
      ];

      // Base events.
      $event_options = [
        'create' => $this->t('On create'),
        'update' => $this->t('On update'),
        'delete' => $this->t('On delete'),
      ];

      // If workflows/content moderation is available, add transitions options.
      if ($has_workflows && $moderation_info && $entity_storage) {
        try {
          // Create a proper unsaved entity with the
          // bundle key from the storage.
          $bundle_key = $entity_storage->getEntityType()->getKey('bundle');
          $fake_entity = $entity_storage->create([$bundle_key => $bundle_id]);

          // Get workflow configured for this entity (may return null).
          $workflow = $moderation_info->getWorkflowForEntity($fake_entity);

          if ($workflow) {
            $plugin = $workflow->getTypePlugin();

            // 1) Try to get initial state passing the entity
            // (plugin may require it).
            try {
              $initial_state = $plugin->getInitialState($fake_entity);
            }
            catch (\InvalidArgumentException $e) {
              // Fallback: read initial state id from workflow config and map
              // to plugin states.
              $initial_state_id = $workflow->get('type_settings')['initial_state'] ?? NULL;
              $states = $plugin->getStates();
              $initial_state = ($initial_state_id && isset($states[$initial_state_id]))
                ? $states[$initial_state_id]
                : NULL;
            }

            // Add "On create - TRANSITION" from the initial
            // state's outgoing transitions.
            if ($initial_state) {
              foreach ($initial_state->getTransitions() as $transition) {
                $key = 'create__' . $transition->id();
                $event_options[$key] = $this->t('On create - @transition', [
                  '@transition' => $transition->label(),
                ]);
              }
            }

            // Add "On update - TRANSITION" for all
            // state transitions (dedupe keys).
            foreach ($plugin->getStates() as $state_id => $state) {
              foreach ($state->getTransitions() as $transition) {
                $key = 'update__' . $state_id . '_' . $transition->id();
                if (!isset($event_options[$key])) {
                  // Include state label to clarify where
                  // ]the transition comes from.
                  $event_options[$key] = $this->t('On update - @transition (from @state)', [
                    '@transition' => $transition->label(),
                    '@state' => $state->label(),
                  ]);
                }
              }
            }
          }
        }
        catch (\Throwable $e) {
          // Do not break the form — log and continue with base options.
          $this->loggerFactory->get('entity_io_webhooks')->warning(
            'Could not load workflow transitions for %bundle: %message',
            ['%bundle' => $bundle_id, '%message' => $e->getMessage()]
          );
        }
      }

      $events = $bundle_config['events'] ?? [];
      $form[$bundle_id]['events'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Events to trigger'),
        '#options' => $event_options,
        '#default_value' => is_array($events) ? $events : [],
      ];

      $form[$bundle_id]['webhooks'] = [
        '#type' => 'details',
        '#title' => $this->t('Webhook settings'),
        '#open' => FALSE,
      ];

      $form[$bundle_id]['webhooks']['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable webhooks for this bundle'),
        '#default_value' => $bundle_config['webhooks']['enabled'] ?? FALSE,
      ];

      $urls = $bundle_config['webhooks']['urls'] ?? [];
      $form[$bundle_id]['webhooks']['urls'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Webhook URLs (one per line)'),
        '#default_value' => is_array($urls) ? implode("\n", $urls) : '',
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[webhooks][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[webhooks][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $headers = $bundle_config['webhooks']['headers'] ?? '';
      $form[$bundle_id]['webhooks']['headers'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Additional headers (JSON or Key: Value per line)'),
        '#default_value' => $headers,
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[webhooks][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[webhooks][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['mail'] = [
        '#type' => 'details',
        '#title' => $this->t('Email notification settings'),
        '#open' => FALSE,
        '#tree' => TRUE,
      ];

      $smtp_url = Url::fromUri('https://www.drupal.org/project/smtp')->toString();
      $smtp_config_url = Url::fromRoute('smtp.config')->toString();
      $form[$bundle_id]['mail']['message'] = [
        '#type' => 'markup',
        '#markup' => $this->t(
          'We use the module <a href=":url">SMTP Authentication Support</a> to send emails. Make sure your site is <a href=":smtp-config">configured</a> to send emails.',
          [':url' => $smtp_url, ':smtp-config' => $smtp_config_url]
        ),
      ];

      $form[$bundle_id]['mail']['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable export mail for this bundle'),
        '#default_value' => $bundle_config['mail']['enabled'] ?? FALSE,
      ];

      $subject = 'Exported entity data – [entity:type]: [entity:label]';
      $form[$bundle_id]['mail']['subject'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Subject'),
        '#default_value' => $bundle_config['mail']['subject'] ?? $subject,
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $body = '
Hello,
<br>
Attached you will find the exported JSON file containing the data for the requested entity.
<br>
<br>
Details:
<br>
<ul>
  <li>Entity Type: [entity:type]</li>
  <li>Entity ID: [entity:id]</li>
  <li>Entity Label: [entity:label]</li>
  <li>Export Date: [current-date:medium]</li>
</ul>
<br>
Best regards,
<br>entity_io Export System
';
      $form[$bundle_id]['mail']['body'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Body Message'),
        '#default_value' => $bundle_config['mail']['body'] ?? $body,
        '#description' => $this->t(
          'You can use tokens. For a list of available tokens, click <a href=":url" target="_blank">here</a>.',
          [':url' => Url::fromUri('https://www.drupal.org/docs/contributed-modules/token/token-user-guide')->toString()]
        ),
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['mail']['email'] = [
        '#type' => 'email',
        '#title' => $this->t('Sent to email'),
        '#default_value' => $bundle_config['mail']['email'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[mail][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['ftp'] = [
        '#type' => 'details',
        '#title' => $this->t('FTP settings'),
        '#open' => FALSE,
        '#tree' => TRUE,
      ];

      $form[$bundle_id]['ftp']['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable export FTP for this bundle'),
        '#default_value' => $bundle_config['ftp']['enabled'] ?? FALSE,
      ];

      $form[$bundle_id]['ftp']['ftp_host'] = [
        '#type' => 'textfield',
        '#title' => $this->t('FTP Host'),
        '#default_value' => $bundle_config['ftp']['ftp_host'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['ftp']['ftp_user'] = [
        '#type' => 'textfield',
        '#title' => $this->t('FTP Username'),
        '#default_value' => $bundle_config['ftp']['ftp_user'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['ftp']['ftp_pass'] = [
        '#type' => 'textfield',
        '#title' => $this->t('FTP Password'),
        '#description' => $this->t('Leave empty to keep the current password.'),
        '#default_value' => $bundle_config['ftp']['ftp_pass'] ?? '',
        '#attributes' => ['type' => 'password'],
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$bundle_id]['ftp']['ftp_path'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Destination Path'),
        '#default_value' => $bundle_config['ftp']['ftp_path'] ?? '',
        '#description' => $this->t('Folder on FTP server where files will be uploaded.'),
        '#states' => [
          'visible' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="' . $bundle_id . '[ftp][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    $form['#tree'] = TRUE;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable('entity_io_webhooks.settings');
    $entity_type_id = $form_state->getValue('entity_type_id');
    $values = $form_state->getValues();

    // Get bundle info to iterate through bundles.
    $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);
    if (empty($bundles)) {
      $bundles = [$entity_type_id => []];
    }

    foreach (array_keys($bundles) as $bundle_id) {
      if (isset($values[$bundle_id]) && is_array($values[$bundle_id])) {
        $bundle_values = $values[$bundle_id];
        $base = "entities.$entity_type_id.$bundle_id";

        $events = isset($bundle_values['events']) ? array_keys(array_filter($bundle_values['events'])) : [];
        $config->set("$base.events", $events);

        $urls_text = $bundle_values['webhooks']['urls'] ?? '';
        $urls = array_filter(array_map('trim', preg_split('/\r?\n/', $urls_text)));
        // --- Save Webhooks ---
        $config->set("$base.webhooks.enabled", !empty($bundle_values['webhooks']['enabled']));
        $config->set("$base.webhooks.urls", $urls);
        $config->set("$base.webhooks.headers", $bundle_values['webhooks']['headers'] ?? '');

        // --- Save Mail ---
        $config->set("$base.mail.enabled", $bundle_values['mail']['enabled'] ?? '');
        $config->set("$base.mail.body", $bundle_values['mail']['body'] ?? '');
        $config->set("$base.mail.subject", $bundle_values['mail']['subject'] ?? '');
        $config->set("$base.mail.email", $bundle_values['mail']['email'] ?? '');

        // --- Save FTP ---
        $config->set("$base.ftp.enabled", $bundle_values['ftp']['enabled'] ?? '');
        $config->set("$base.ftp.ftp_host", $bundle_values['ftp']['ftp_host'] ?? '');
        $config->set("$base.ftp.ftp_user", $bundle_values['ftp']['ftp_user'] ?? '');
        $config->set("$base.ftp.ftp_pass", $bundle_values['ftp']['ftp_pass'] ?? '');
        $config->set("$base.ftp.ftp_path", $bundle_values['ftp']['ftp_path'] ?? '');
      }
    }

    $config->save();
    parent::submitForm($form, $form_state);
  }

}
