<?php

namespace Drupal\entity_io_webhooks\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Psr\Log\LoggerInterface;

/**
 * Service to send emails with entity JSON attachments.
 */
class EntityIoMailSender {

  /**
   * Mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs an EntityIoMailSender object.
   */
  public function __construct(MailManagerInterface $mailManager, EntityTypeManagerInterface $entityTypeManager, ConfigFactoryInterface $configFactory, LoggerInterface $logger, AccountProxyInterface $current_user) {
    $this->mailManager = $mailManager;
    $this->entityTypeManager = $entityTypeManager;
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->currentUser = $current_user;
  }

  /**
   * Send an email with the entity JSON attachment.
   */
  public function trigger(EntityInterface $entity, string $path_json, string $operation, string $to, string $subject, string $bodyText) {
    // Get 'from' email from SMTP config, fallback to site mail.
    $smtp_config = $this->configFactory->get('smtp.settings');
    $smtp_from = $smtp_config ? $smtp_config->get('smtp_from') : NULL;
    $system_config = $this->configFactory->get('system.site');
    $from = !empty($smtp_from) ? $smtp_from : ($system_config ? $system_config->get('mail') : NULL);

    $reply_to = $smtp_config ? $smtp_config->get('reply_to') : NULL;
    if (empty($reply_to)) {
      $reply_to = $from;
    }

    $params = [
      'subject' => $subject,
      'body' => '<html><body>' . strtr($bodyText, [
        '@operation' => $operation,
        '@id' => $entity->id(),
        '@type' => $entity->getEntityTypeId(),
        '@label' => $entity->label(),
      ]) . '</body></html>',
      'attachments' => [
        [
          'filepath' => $path_json,
          'filename' => basename($path_json),
          'filemime' => 'application/json',
        ],
      ],
      'reply-to' => $reply_to,
      'html' => TRUE,
    ];

    $langcode = $this->currentUser->getPreferredLangcode();

    $result = $this->mailManager->mail(
      'entity_io_webhooks',
      'entity_notify',
      $to,
      $langcode,
      $params,
      $from,
      TRUE
    );

    if (empty($result) || empty($result['result'])) {
      $this->logger->error('Failed to send entity export email for entity @id', ['@id' => $entity->id()]);
    }
    else {
      $this->logger->info('Entity export email sent for entity @id', ['@id' => $entity->id()]);
    }
  }

}
