<?php

namespace Drupal\entity_io\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for displaying entity IO logs.
 */
class EntityIoLogController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * Constructor.
   */
  public function __construct(Connection $database, DateFormatterInterface $date_formatter, FormBuilderInterface $form_builder) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('form_builder')
    );
  }

  /**
   * Log page.
   */
  public function logPage(Request $request) {

    $build['filter_form'] = $this->formBuilder->getForm('Drupal\entity_io\Form\EntityExportLogFilterForm');

    $header = [
      'lid' => 'ID',
      'uid' => 'User',
      'operation' => 'Operation',
      'entity_type' => 'Type',
      'entity_id' => 'Entity ID',
      'status' => 'Status',
      'message' => 'Message',
      'created' => 'Date',
    ];

    $filters = $request->query->all();
    $items_per_page = $filters['items_per_page'] ?? 100;

    $query = $this->database->select('entity_io_log', 'l')
      ->fields('l')
      ->orderBy('created', 'DESC');

    // Apply filters from the request.
    if (!empty($filters['lid'])) {
      $query->condition('l.lid', $filters['lid']);
    }
    if (!empty($filters['uid'])) {
      $query->condition('l.uid', $filters['uid']);
    }
    if (!empty($filters['operation'])) {
      $query->condition('l.operation', $filters['operation']);
    }
    if (!empty($filters['entity_type'])) {
      $query->condition('l.entity_type', $filters['entity_type']);
    }
    if (!empty($filters['entity_id'])) {
      $query->condition('l.entity_id', $filters['entity_id']);
    }
    if (!empty($filters['status'])) {
      $query->condition('l.status', $filters['status']);
    }
    if (!empty($filters['message'])) {
      $query->condition('l.message', '%' . $this->database->escapeLike($filters['message']) . '%', 'LIKE');
    }

    // Add pager.
    $paged_query = $query->extend(PagerSelectExtender::class)
      ->limit($items_per_page);

    $rows = [];
    foreach ($paged_query->execute() as $record) {
      $rows[] = [
        'lid' => $record->lid,
        'uid' => $record->uid,
        'operation' => $record->operation,
        'entity_type' => $record->entity_type,
        'entity_id' => $record->entity_id,
        'status' => $record->status,
        'message' => $record->message,
        'created' => $this->dateFormatter->format($record->created, 'short'),
      ];
    }

    $build['log_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No logs found.'),
    ];

    $build['pager'] = ['#type' => 'pager'];

    return $build;
  }

}
