<?php

namespace Drupal\entity_io\Form\Export;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\TermInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Taxonomy JSON export form.
 */
class TaxonomyJsonExportForm extends FormBase {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The class resolver service.
   *
   * @var \Drupal\Core\DependencyInjection\ClassResolverInterface
   */
  protected $classResolver;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * Constructs the form.
   */
  public function __construct(LanguageManagerInterface $language_manager, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, ClassResolverInterface $class_resolver, $exporter) {
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->classResolver = $class_resolver;
    $this->exporter = $exporter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('class_resolver'),
      $container->get('entity_io.exporter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $taxonomy_term = NULL) {
    $language = $this->languageManager->getCurrentLanguage();

    if (!$taxonomy_term instanceof TermInterface) {
      return ['error' => ['#markup' => 'Invalid taxonomy term']];
    }

    if ($taxonomy_term->hasTranslation($language->getId())) {
      $taxonomy_term = $taxonomy_term->getTranslation($language->getId());
    }

    $config = $this->configFactory->get('entity_io.taxonomy_term_field_settings');

    // Get base fields using injected services.
    $entity_type = $this->entityTypeManager->getDefinition('taxonomy_term');
    $base_fields = Term::baseFieldDefinitions($entity_type);
    // Use injected entity_field.manager to fetch field definitions.
    $base_fields = $this->entityFieldManager->getFieldDefinitions('taxonomy_term', $entity_type->id());

    $form[$taxonomy_term->bundle()]['base_fields'] = [
      '#type' => 'details',
      '#title' => 'Base Fields',
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      if (!str_starts_with($base_field_name, 'field_')) {
        $default = $config->get($taxonomy_term->bundle())['base_fields'][$base_field_name] ?? '';
        $form[$taxonomy_term->bundle()]['base_fields'][$base_field_name] = [
          '#type' => 'checkbox',
          '#title' => $base_field_name,
          '#default_value' => $default,
        ];
      }
    }

    // Load all fields for this taxonomy vocabulary using
    // injected entityTypeManager.
    $fields = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
      'entity_type' => 'taxonomy_term',
      'bundle' => $taxonomy_term->bundle(),
    ]);

    if ($fields) {

      // Custom fields.
      $form['fields'] = [
        '#type' => 'details',
        '#title' => 'Fields',
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      // Use only the field object; key variable was unused previously.
      foreach ($fields as $field) {
        $default = $config->get($taxonomy_term->bundle())['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->getName(),
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => 'Check All',
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => 'Clear All',
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => 'Toggle',
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $options = [];
    for ($i = -1; $i <= 100; $i++) {
      $options[(string) $i] = (string) $i;
    }

    $form['depth'] = [
      '#type' => 'select',
      '#title' => 'Entity reference depth',
      '#options' => $options,
      '#default_value' => $config->get($taxonomy_term->bundle())['depth'] ?? -1,
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['export'] = [
      '#type' => 'submit',
      '#value' => 'Export as JSON',
    ];

    $form['queue_export'] = [
      '#type' => 'button',
      '#value' => $this->t('Add to Export Queue'),
      '#ajax' => [
        'callback' => '::ajaxQueueExport',
        'wrapper' => 'queue-export-messages',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding to queue...'),
        ],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['queue-export-btn'],
      ],
    ];

    $form['queue_messages'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $taxonomy_term = $this->getRequest()->attributes->get('taxonomy_term');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    if ($taxonomy_term instanceof TermInterface) {
      $exporter = $this->exporter;
      $max_depth = (int) $form_state->getValue('depth');
      $exporter::toJson($taxonomy_term, 0, $max_depth, $selected_fields, $current_langcode);
      $json = $exporter::$json ?? json_encode([]);
      $fileName = $exporter::$fileName ?? 'export.json';

      $response = new Response($json);
      $response->headers->set('Content-Type', 'application/json');
      $response->headers->set('Content-Disposition', 'attachment; filename="' . $fileName . '"');
      $response->send();
      die;
    }
  }

  /**
   * AJAX callback for queue export button.
   */
  public function ajaxQueueExport(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields_bundle = $form_state->getValue($this->getRequest()->attributes->get('taxonomy_term')->bundle())['base_fields'] ?? [];
    $base_fields = array_filter($base_fields_bundle);
    $max_depth = (int) $form_state->getValue('depth');
    $taxonomy_term = $this->getRequest()->attributes->get('taxonomy_term');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    $response = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    if ($taxonomy_term instanceof TermInterface) {
      try {
        $data = [
          'entity_type' => $taxonomy_term->getEntityTypeId(),
          'entity_id' => $taxonomy_term->id(),
          'depth' => 0,
          'max_depth' => $max_depth,
          'selected_fields' => $selected_fields,
          'langcode' => $current_langcode,
          'is_revision' => FALSE,
        ];

        $controller = $this->classResolver->getInstanceFromDefinition('Drupal\entity_io_queue\Controller\EntityIoQueueController');
        $result = $controller->addExportItem($data);

        if ($result) {
          $response['message'] = [
            '#markup' => '<div class="messages messages--status">' .
            $this->t('Taxonomy term @title has been added to the export queue.', [
              '@title' => $taxonomy_term->label(),
            ]) . '</div>',
          ];
        }
        else {
          $response['message'] = [
            '#markup' => '<div class="messages messages--error">' .
            $this->t('Failed to add taxonomy term to queue.') . '</div>',
          ];
        }
      }
      catch (\Exception $e) {
        $response['message'] = [
          '#markup' => '<div class="messages messages--error">' .
          $this->t('Failed to add taxonomy term to queue: @message', [
            '@message' => $e->getMessage(),
          ]) . '</div>',
        ];
      }
    }
    else {
      $response['message'] = [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('Invalid taxonomy_term entity.') . '</div>',
      ];
    }

    return $response;
  }

}
