<?php

namespace Drupal\entity_io\Helper;

/**
 * Utilities to compare exported JSON structures against actual entities.
 *
 * Produces a structured diff tree suitable for rendering or inspection.
 */
class EntityJsonDiff {

  /**
   * Cached exported entities map (uuid => exported data).
   *
   * @var array
   */
  private static $exportedEntities = [];

  /**
   * Tracks already compared entities to avoid infinite recursion.
   *
   * @var array
   */
  private static $alreadyCompared = [];

  /**
   * Compares an exported JSON representation against the corresponding entity.
   *
   * - Handles simple fields and multi-value fields.
   * - For entity_reference(_revisions): if JSON embeds ['entity'],
   *   - compares recursively.
   * - Returns a nested tree of differences so you can render as HTML.
   *
   * @param array $json
   *   Parsed JSON array representing the exported entity data.
   * @param string $scope
   *   Which fields to compare: 'all' (default), 'base' (core/basic fields),
   *   or 'admin' (fields added by administrator, i.e. starting with "field_").
   *
   * @return array
   *   Structured diff array:
   *   [
   *     'entity_' => 'node:UUID (lang)',
   *     'fields' => [
   *        'field_name' => [
   *          'field' => 'field_name',
   *          'entity_value' => <entity field value>,
   *          'json_value' => <json value>,
   *          'children' => [ ... recursive child diffs ... ] (optional)
   *        ],
   *        ...
   *     ],
   *   ]
   */
  public static function compareEntity(array $json, string $scope = 'all'): array {
    if (isset($json['exportedEntities'])) {
      self::$exportedEntities = $json['exportedEntities'];
    }

    $langcode = $json['langcode'][0]['value'] ?? \Drupal::languageManager()->getDefaultLanguage()->getId();

    // Parse entity info (type, bundle, uuid, etc.) from the JSON structure.
    $entityData = JsonParseEntityData::parseEntityInfo($json);
    extract($entityData);

    // Attempt to load the entity by UUID via the entity storage.
    $entity_storage = \Drupal::service('entity_type.manager')
      ->getStorage($json['__entity_type__']);
    $loaded = $entity_storage->loadByProperties(['uuid' => $uuid]);
    $entity = reset($loaded);

    if (!$entity) {
      $bind = \Drupal::service('entity_io.storage')->get($uuid);
      if (!$entity && $bind) {
        $entity = EntityLoader::loadEntity($bind);
      }
    }

    if ($entity && $entity->hasTranslation($langcode)) {
      $entity = $entity->getTranslation($langcode);
    }

    if (isset(self::$alreadyCompared[$uuid]) && in_array($langcode, self::$alreadyCompared[$uuid], TRUE)) {
      // Avoid infinite recursion on circular references.
      return [
        'fields' => [],
      ];
    }

    self::$alreadyCompared[$uuid][] = $langcode;

    $entity_type = $entity ? $entity->getEntityTypeId() : '';
    $uuid = $entity ? $entity->uuid() : '';
    $entity_label = $entity ? (ucwords($entity_type) . ':' . $uuid) : NULL;

    $diffs = [
      'entity_' => $entity_label,
      'fields' => [],
    ];

    // Get field definitions for this entity type/bundle.
    $definitions = EntityFields::get($type, $bundle);
    $baseDefinitions = EntityFields::getBaseFields($type, $bundle);

    // Filter definitions according to requested scope.
    if ($scope === 'all') {
      $definitions = array_merge($baseDefinitions, $definitions ?? []);
    }
    elseif ($scope === 'base') {
      $definitions = $baseDefinitions;
    }

    foreach ($definitions as $field_name => $definition) {
      // Remove large binary content from JSON before comparing.
      if (isset($json[$field_name][0]['base64'])) {
        unset($json[$field_name][0]['base64']);
      }

      /** @var \Drupal\Core\Field\FieldItemListInterface|null $entity_field */
      $entity_field = $entity ? $entity->get($field_name) : NULL;
      $type = $definition->getType();

      // Get entity field value via exporter helper.
      $entity_targets = $entity_field ? \Drupal::service('entity_io.export')::getFieldValue($entity_field) : '';

      $json_targets = $json[$field_name] ?? NULL;

      if (isset($entity_targets[0]['base64'])) {
        unset($entity_targets[0]['base64']);
      }

      // Build field diff structure (split over multiple lines for readability).
      $diffs['fields'][$field_name] = [
        'field' => $field_name,
        'entity_value' => $entity_targets,
        'json_value' => $json_targets,
      ];

      if (isset($json[$field_name][0]['target__uuid'])) {
        $target_uuid = $json[$field_name][0]['target__uuid'];
        // dd($target_uuid, self::$exportedEntities);.
        if (isset(self::$exportedEntities[$target_uuid][$langcode])) {
          $json[$field_name][0] = self::$exportedEntities[$target_uuid][$langcode];
        }
      }

      // If the field references an embedded entity (image/file), recurse.
      if ($type === 'image' && !empty($json[$field_name][0])) {
        $child = self::compareEntity($json[$field_name][0], $scope);
        $diffs['fields'][$field_name]['children'] = $child;
      }
      elseif ($type === 'file' && !empty($json[$field_name][0])) {
        $child = self::compareEntity($json[$field_name][0], $scope);
        $diffs['fields'][$field_name]['children'] = $child;
      }
      // For references, compare IDs and recurse if
      // JSON embeds the referenced entity.
      elseif (in_array($type, ['entity_reference', 'entity_reference_revisions'], TRUE)
        && !in_array($field_name, ['type', 'roles', 'bundle'])) {

        $target_type = $definition->getSetting('target_type');

        if (!empty($json[$field_name]) && is_array($json[$field_name][0])
          && !in_array($target_type, ['media_type', 'node_type', 'comment_type', 'block_content_type'])) {

          // Skip simple ID-only references.
          if (isset($json[$field_name][0]['target_id']) || $field_name === 'entity_id' || $field_name === 'vid') {
            continue;
          }

          // If JSON embeds the referenced entity, recurse directly.
          if (!empty($json[$field_name][0])) {
            $child = self::compareEntity($json[$field_name][0], $scope);
            $diffs['fields'][$field_name]['children'] = $child;
          }
        }
      }
    }

    return $diffs;
  }

  /**
   * Convenience: produce a structured diff for given exported JSON.
   *
   * Returns an empty array when no meaningful fields/children were found.
   *
   * @param array $json
   *   Parsed JSON array representing the exported entity data.
   * @param string $scope
   *   Which fields to compare: 'all' (default), 'base', or 'admin'.
   *
   * @return array
   *   Structured diffs or empty array.
   */
  public static function diff(array $json, string $scope = 'all') {
    $diffs = self::compareEntity($json, $scope);

    // If no differences or children, return empty to indicate "no diff".
    if (empty($diffs['fields']) && empty($diffs['children'])) {
      return [];
    }
    return $diffs;
  }

  /**
   * Convenience wrapper to compare only base (non field_*) fields.
   */
  public static function diffBaseFields(array $json) {
    return self::diff($json, 'base');
  }

  /**
   * Convenience wrapper to compare only (field_*) fields.
   */
  public static function diffFields(array $json) {
    return self::diff($json, 'fields');
  }

}
