<?php

namespace Drupal\entity_io\Helper;

use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * JSON validation and comparison helper.
 */
class JsonValidate {

  /**
   * Stores log messages for the last validation run.
   *
   * @var array
   */
  protected static $log = [];

  /**
   * Stores validation errors for the last validation run.
   *
   * @var array
   */
  protected static $errors = [];

  /**
   * Tracks used entities to prevent recursion.
   *
   * @var array
   */
  protected static $usedEntities = [];

  /**
   * Resets logs/errors.
   */
  public static function reset(): void {
    self::$log = [];
    self::$errors = [];
  }

  /**
   * Validates JSON entity data against required definitions.
   *
   * @param array $json
   *   The JSON data for the entity.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public static function validateEntity(array $json): bool {
    $entityData = JsonParseEntityData::parseEntityInfo($json);
    extract($entityData);

    // First validation to know if the entity is valid.
    if (!$type || !$bundle || !$uuid) {
      die("invalid JSON entity. Type, bundle, uuid required!");
    }
    $entity_id_info = $bundle . (isset($json['langcode'][0]['value']) ? '(' . $json['langcode'][0]['value'] . ')' : '');
    $entityFields = EntityFields::getAll($type, $bundle);
    // $required_fields = EntityFields::getRequired($type, $bundle); // unused
    $valid = TRUE;

    if (!self::validateFields($json, $entityFields, $entity_id_info)) {
      $valid = FALSE;
    }

    if (!empty($json['translations']) && is_array($json['translations'])) {
      foreach ($json['translations'] as $langcode) {
        $translation_data = $json['exportedEntities'][$json['uuid'][0]['value']][$langcode];
        $translation_id_info = $entity_id_info . " ($langcode)";
        if (!self::validateFields($translation_data, $entityFields, $translation_id_info)) {
          $valid = FALSE;
        }
      }
    }

    return $valid;
  }

  /**
   * Validates fields for a single entity/translation and recurses references.
   *
   * @param array $json
   *   The JSON data for the entity/translation.
   * @param array $fields
   *   Associative array: field_name => FieldDefinitionInterface.
   * @param string $entity_id_info
   *   Entity identifier string for error context.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  protected static function validateFields(array $json, array $fields, string $entity_id_info): bool {
    $valid = TRUE;

    foreach ($fields as $field_name => $field_definition) {
      $field_type = $field_definition->getType();
      if (!$field_definition instanceof FieldDefinitionInterface) {
        self::$errors[] = [
          'entity' => $entity_id_info,
          'field' => $field_name,
          'message' => 'Invalid field definition object.',
        ];
        $valid = FALSE;
        continue;
      }

      // Skip base fields that do not need validation.
      if ($field_definition->getFieldStorageDefinition()->isBaseField()) {
        continue;
      }

      if (!self::validateFieldByType($field_type, $json[$field_name] ?? NULL, $field_definition, $field_name)) {
        $valid = FALSE;
        break;
      }
    }

    return $valid;
  }

  /**
   * Validates a field based on the field type.
   *
   * @param string $type
   *   The field type.
   * @param mixed $value
   *   The field value.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $definition
   *   The field definition.
   * @param string $field_label
   *   The field label.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public static function validateFieldByType(string $type, $value, FieldDefinitionInterface $definition, string $field_label): bool {
    $valid = TRUE;
    $field_type = $definition->getType();
    $settings = $definition->getSettings();
    $field_name = $definition->getName();

    // Check if required.
    if ($value === NULL) {
      $config = \Drupal::service('config.factory')->get('entity_io.import_settings');
      if (!$config->get('ignore_required_validation_import')) {
        if ($definition->isRequired()) {
          self::$errors[] = t('@field it is mandatory, but it was not informed.', ['@field' => $field_label]);
          return FALSE;
        }
      }
      return TRUE;
    }

    // Normalize value to array of items.
    $items = is_array($value) && isset($value[0]) ? $value : [$value];
    $settings = $definition->getSettings();

    // Check cardinality (max number of values allowed).
    $cardinality = $definition->getFieldStorageDefinition()->getCardinality();
    if ($cardinality > 0 && count($items) > $cardinality) {
      self::$errors[] = t('@field: exceeds the allowed cardinality (@allowed).', [
        '@field' => $field_label,
        '@allowed' => $cardinality,
      ]);
      $valid = FALSE;
    }

    foreach ($items as $delta => $item) {
      switch ($field_type) {
        /* ----------------- STRING & TEXT TYPES ----------------- */
        case 'string':
        case 'string_long':
        case 'text':
        case 'text_long':
        case 'text_with_summary':
          $val = $item['value'] ?? '';
          if (!is_string($val)) {
            self::$errors[] = t('@field[#@delta]: Must be a string.', ['@field' => $field_label, '@delta' => $delta]);
            $valid = FALSE;
          }
          elseif (!empty($settings['max_length']) && strlen($val) > $settings['max_length']) {
            self::$errors[] = t('@field[#@delta]: Exceeds the maximum length of @max characters.', [
              '@field' => $field_label,
              '@delta' => $delta,
              '@max' => $settings['max_length'],
            ]);
            $valid = FALSE;
          }
          break;

        /* ----------------- METATAGS ----------------- */
        case 'metatag':
          if ($item && !is_array($item)) {
            self::$errors[] = t('@field[#@delta]: Must be an associative array of meta tags.', [
              '@field' => $field_label,
              '@delta' => $delta,
            ]);
            $valid = FALSE;
          }
          break;

        /* ----------------- EMAIL ----------------- */
        case 'email':
          if ($item) {
            $val = trim($item['value'] ?? '');
            if (!filter_var($val, FILTER_VALIDATE_EMAIL)) {
              self::$errors[] = t('@field[#@delta]: Invalid email "@value".', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@value' => $val,
              ]);
              $valid = FALSE;
            }
          }
          break;

        /* ----------------- INTEGER & LIST_INTEGER ----------------- */
        case 'integer':
        case 'list_integer':
          if (isset($item['value']) && $item['value']) {
            $val = $item['value'] ?? NULL;
            if (!is_numeric($val) || intval($val) != $val) {
              self::$errors[] = t('@field[#@delta]: Must be an integer.', ['@field' => $field_label, '@delta' => $delta]);
              $valid = FALSE;
            }
            if (isset($settings['min']) && $val < $settings['min']) {
              self::$errors[] = t('@field[#@delta]: Minimum allowed value is @min.', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@min' => $settings['min'],
              ]);
              $valid = FALSE;
            }
            if (isset($settings['max']) && $val > $settings['max']) {
              self::$errors[] = t('@field[#@delta]: Maximum allowed value is @max.', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@max' => $settings['max'],
              ]);
              $valid = FALSE;
            }
          }
          break;

        case 'comment':
          if ($item && !is_numeric($item['cid'][$delta]['value'] ?? NULL)) {
            self::$errors[] = t('Invalid comment reference in @label at delta @delta.', [
              '@label' => $label,
              '@delta' => $delta,
            ]);
            $valid = FALSE;
          }
          break;

        /* ----------------- LIST_STRING ----------------- */
        case 'list_string':
          if ($item) {
            $val = $item['value'] ?? '';
            $allowed = array_keys($settings['allowed_values'] ?? []);
            if (!in_array($val, $allowed)) {
              self::$errors[] = t(
                '@field[#@delta]: Invalid value "@value". Allowed: @allowed.',
                [
                  '@field' => $field_label,
                  '@delta' => $delta,
                  '@value' => $val,
                  '@allowed' => implode(', ', $allowed),
                ]
              );
              $valid = FALSE;
            }
          }
          break;

        /* ----------------- BOOLEAN ----------------- */
        case 'boolean':
          if (isset($item['value'])) {
            if (!in_array($item['value'], [0, 1, '0', '1', TRUE, FALSE])) {
              self::$errors[] = t('@field[#@delta]: Must be boolean (0 ou 1).', [
                '@field' => $field_label,
                '@delta' => $delta,
              ]);
              $valid = FALSE;
            }
          }
          break;

        /* ----------------- DECIMAL ----------------- */
        case 'decimal':
          if (isset($item['value']) && $item['value']) {
            $val = $item['value'] ?? NULL;
            if (!is_numeric($val)) {
              self::$errors[] = t('@field[#@delta]: Must be a decimal number.', [
                '@field' => $field_label,
                '@delta' => $delta,
              ]);
              $valid = FALSE;
            }
            else {
              $precision = $settings['precision'] ?? 10;
              $scale = $settings['scale'] ?? 2;
              $parts = explode('.', (string) $val);
              $integer_length = strlen($parts[0]);
              $decimal_length = isset($parts[1]) ? strlen($parts[1]) : 0;
              if ($integer_length + $decimal_length > $precision || $decimal_length > $scale) {
                self::$errors[] = t('@field[#@delta]: Number exceeds precision (@precision) or scale (@scale).', [
                  '@field' => $field_label,
                  '@delta' => $delta,
                  '@precision' => $precision,
                  '@scale' => $scale,
                ]);
                $valid = FALSE;
              }
            }
          }

          break;

        /* ----------------- FLOAT ----------------- */
        case 'float':
        case 'list_float':
          if (isset($item['value']) && $item['value']) {
            $val = $item['value'] ?? NULL;
            if (!is_numeric($val)) {
              self::$errors[] = t('@field[#@delta]: Must be a float number.', [
                '@field' => $field_label,
                '@delta' => $delta,
              ]);
              $valid = FALSE;
            }
          }
          break;

        case 'telephone':
          if (isset($item['value']) && $item['value']) {
            $value = $item['value'] ?? NULL;
            if (!preg_match('/^[0-9\-\+\(\) ]+$/', $value)) {
              self::$errors[] = t('@field[#@delta]: Enter a valid telephone number.', [
                '@field' => $field_label,
                '@delta' => $delta,
              ]);
              $valid = FALSE;
            }

            // Exemplo extra: exigir no mínimo 8 dígitos numéricos.
            $digits = preg_replace('/\D/', '', $value);
            if (strlen($digits) < 8) {
              self::$errors[] = t('@field[#@delta]: Must be at least 8 digits.', [
                '@field' => $field_label,
                '@delta' => $delta,
              ]);
              $valid = FALSE;
            }
          }
          break;

        /* ----------------- DATETIME ----------------- */
        case 'datetime':
        case 'timestamp':
          if ($item) {

            $val = trim((string) ($item['value'] ?? ''));
            if (is_numeric($val)) {
              $val = date('Y-m-d\TH:i:s', (int) $val);
            }
            $dt = \DateTime::createFromFormat('Y-m-d\TH:i:s', $val);
            if (!$dt) {
              self::$errors[] = t('@field[#@delta]: Invalid date "@value".', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@value' => $val,
              ]);
              $valid = FALSE;
            }
          }
          break;

        case 'daterange':
          if (isset($item['value']) && $item['value'] && isset($item['end_value']) && $item['end_value']) {
            $start_date = $item['value'] ?? NULL;
            $end_date = $item['end_value'] ?? NULL;

            if ($start_date && $end_date && strtotime($end_date) < strtotime($start_date)) {
              self::$errors[] = t('@field[#@delta]: Invalid date range "@start" - "@end".', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@start' => $start_date,
                '@end' => $end_date,
              ]);
              $valid = FALSE;
            }
          }
          break;

        /* ----------------- LINK ----------------- */
        case 'link':
          $uri = $item['uri'] ?? '';
          $title = $item['title'] ?? '';
          if ($uri) {
            if (!preg_match('/^(internal:\/|entity:|route:)/', $uri) && !filter_var($uri, FILTER_VALIDATE_URL)) {
              self::$errors[] = t('@field[#@delta]: Invalid URL "@uri".', [
                '@field' => $field_label,
                '@delta' => $delta,
                '@uri' => $uri,
              ]);
              $valid = FALSE;
            }
          }
          if (!empty($title) && !is_string($title)) {
            self::$errors[] = t('@field[#@delta]: Title is not a valid string.', [
              '@field' => $field_label,
              '@delta' => $delta,
            ]);
            $valid = FALSE;
          }
          break;

        /* ----------------- ENTITY REFERENCES ----------------- */
        case 'file':
        case 'image':
        case 'entity_reference':
        case 'entity_reference_revisions':
          // Prevent infinite recursion.
          if (isset($item['target__uuid'])) {
            $valid = TRUE;
            continue 2;
          }
          if (is_array($item) && count($item) && !isset($item['__entity_type__']) && !isset($item['target_id'])) {
            self::$errors[] = t('@field[#@delta]: Invalid reference (missing __entity_type__ or target_id).', [
              '@field' => $field_label,
              '@delta' => $delta,
            ]);
            $valid = FALSE;
          }
          elseif (!isset($item['__entity_type__']) && isset($item['target_id']) && !$item['target_id']) {
            // Only referenced by ID, there is no way to validate
            // the referenced entity.
            // You could try to load the entity from the bank,
            // but that's out of scope.
            self::$errors[] = t('@field[#@delta]: Reference by ID (target_id) without embedded entity; validation ignored.', [
              '@field' => $field_label,
              '@delta' => $delta,
            ]);
            $valid = FALSE;
          }
          break;

        default:
          $module_handler = \Drupal::moduleHandler();
          $hook_specific = 'entity_io_validate_' . $field_type . '_alter';
          $errorMessage = '';

          if ($module_handler->hasImplementations($hook_specific)) {
            $module_handler->invokeAll($hook_specific, [
              &$valid,
              &$errorMessage,
              $item,
              $field_name,
              $field_type,
              $delta,
            ]);
            if ($errorMessage) {
              self::$errors[] = $errorMessage;
            }

          }
          else {
            \Drupal::service('entity_io.logger')->log('validation', NULL, NULL, 'error', t('Validation not implemented for field type: @type.', ['@type' => $type]));
          }
      }
    }

    return $valid;
  }

  /**
   * Gets the log messages.
   *
   * @return array
   *   Log messages.
   */
  public static function getLog(): array {
    return self::$log;
  }

  /**
   * Gets the validation errors.
   *
   * @return array
   *   Validation errors.
   */
  public static function getErrors() {
    return self::$errors;
  }

}
