<?php

namespace Drupal\entity_io\Service;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Url;

/**
 * Service to save exported entity JSON to a file.
 */
final class EntityIoExporter {

  /**
   * Public URL to the exported file.
   *
   * @var string
   */
  public static $publicUrl = '';
  /**
   * Public path to the exported file.
   *
   * @var string
   */
  public static $publicPath = '';
  /**
   * Public path including filename to the exported file.
   *
   * @var string
   */
  public static $publicPathFile = '';
  /**
   * Real file system path to the exported file.
   *
   * @var string
   */
  public static $publicRealPathFile = '';
  /**
   * JSON content of the exported entity.
   *
   * @var string
   */
  public static $json = '';
  /**
   * Storage scheme (public/private).
   *
   * @var string
   */
  public static $scheme = '';
  /**
   * Directory path for exports.
   *
   * @var string
   */
  public static $directory = '';
  /**
   * Filename of the exported file.
   *
   * @var string
   */
  public static $fileName = '';

  /**
   * Export entity to JSON file and save to configured storage.
   *
   * This method exports an entity to JSON format, creates the necessary
   * directory structure, saves the file, and returns the public URL.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to export.
   * @param int $depth
   *   Current depth level for entity references.
   * @param int $max_depth
   *   Maximum depth to traverse entity references.
   * @param array $selected_fields
   *   Array of field names to include in export.
   * @param string $current_langcode
   *   Language code for the export.
   * @param bool $isRevision
   *   Whether this is a revision export.
   *
   * @return string
   *   The public URL to the exported file.
   *
   * @throws \Exception
   *   When directory creation fails.
   */
  public static function toJson($entity, $depth, $max_depth, $selected_fields, $current_langcode, $isRevision = FALSE) {
    $data = \Drupal::service('entity_io.export')::export($entity, $depth, $max_depth, $selected_fields, $current_langcode);

    $module_handler = \Drupal::service('extension.list.module');
    $module = $module_handler->getExtensionInfo('entity_io');
    $data['__version__'] = $module['version'] ?? 'unknown';

    $json = json_encode([$data]);

    $entity_type = $entity->getEntityTypeId();
    $fileName = $entity_type . '-' . $entity->id() . '.json';
    if ($isRevision) {
      $fileName = $entity_type . '-' . $entity->id() . '-rev-' . $entity->getRevisionId() . '.json';
    }

    $config = \Drupal::config('entity_io.export_settings');
    $scheme = $config->get('storage_scheme') ?? 'public';
    $directory = $config->get('directory') ?? 'entity_io_exports';

    self::$scheme = $scheme;
    self::$directory = $directory;
    self::$publicPath = $scheme . '://' . $directory . '/' . $entity_type;

    // Make sure the directory exists.
    \Drupal::service('file_system')->prepareDirectory(self::$publicPath, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    // Generate a unique file name (keeping the entity ID
    // and incrementing after it).
    $fileName = self::getUniqueFileName(self::$publicPath, $fileName);

    self::$fileName = $fileName;
    self::$publicPathFile = self::$publicPath . '/' . $fileName;
    self::$publicRealPathFile = \Drupal::service('file_system')->realpath(self::$publicPathFile);

    if (!is_dir(self::$publicPath)) {
      throw new \Exception('Failed to create directory: ' . self::$publicPath);
    }

    self::$publicUrl = \Drupal::service('file_url_generator')->generateAbsoluteString(self::$publicPathFile);
    self::$json = $json;

    file_put_contents(self::$publicRealPathFile, $json);

    // If private scheme, generate a route URL for downloading.
    if ($scheme === 'private') {
      self::$publicUrl = Url::fromRoute('entity_io.private_file_download', [
        'entity_type' => $entity_type,
        'filename' => $fileName,
      ])->setAbsolute(TRUE)->toString();
    }

    return self::$publicUrl;
  }

  /**
   * Returns a unique file name, keeping the entity ID.
   */
  protected static function getUniqueFileName($directory, $fileName) {
    // $directory: ex "public://entity_io_exports/node"
    // $fileName: ex "taxonomy_term-56.json" ou "taxonomy_term-56-rev-8.json"
    $file_system = \Drupal::service('file_system');

    $info = pathinfo($fileName);
    // Sem extensão.
    $basename = $info['filename'];
    $ext = isset($info['extension']) ? '.' . $info['extension'] : '';

    // Original Path to check existence.
    $originalPath = $directory . '/' . $basename . $ext;
    $originalReal = $file_system->realpath($originalPath);

    // If the original file does NOT exist, return the original name.
    if (!$originalReal || !file_exists($originalReal)) {
      return $fileName;
    }

    // Extract the "base" that should be preserved:
    // - captures: "<type>-<id>" or "<type>-<id>-rev-<revId>"
    // - ignores any additional incrementers already present (ex: -1-2-3)
    // regex: ^(.+?-\d+(?:-rev-\d+)?)(?:-\d+)*$.
    if (preg_match('/^(.+?-\d+(?:-rev-\d+)?)(?:-\d+)*$/', $basename, $matches)) {
      $base = $matches[1];
    }
    else {
      // fallback: if not matched (rare case), use
      // basename without removing anything.
      $base = $basename;
    }

    // Try incrementers starting from 1 until a free name is found.
    $counter = 1;
    while (TRUE) {
      $candidate = $base . '-' . $counter . $ext;
      $candidatePath = $directory . '/' . $candidate;
      $candidateReal = $file_system->realpath($candidatePath);

      if (!$candidateReal || !file_exists($candidateReal)) {
        // Found free name.
        return $candidate;
      }

      $counter++;

      // (Optional) protect against infinite loop - high limit (ex: 10000).
      if ($counter > 10000) {
        throw new \Exception('Unable to find unique file name after 10000 attempts for base: ' . $base);
      }
    }
  }

}
