<?php

namespace Drupal\entity_io_push\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\entity_io_push\Service\JsonPushService;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to deploy an entity JSON to a selected server.
 */
class DeployEntityUserForm extends FormBase {

  /**
   * The JsonPushService.
   *
   * @var \Drupal\entity_io_push\Service\JsonPushService
   */
  protected $pushService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The field config storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $fieldConfigStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity IO export service.
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The entity being deployed.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $entity;

  /**
   * Constructs a DeployEntityUserForm object.
   */
  public function __construct(
    JsonPushService $push_service,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    $entity_io_export,
  ) {
    $this->pushService = $push_service;
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->fieldConfigStorage = $entity_type_manager->getStorage('field_config');
    $this->languageManager = $language_manager;
    $this->entityIoExport = $entity_io_export;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_io_push.json_push_service'),
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('entity_io.export'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_io_push_deploy_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
    ?EntityInterface $node = NULL,
    ?EntityInterface $comment = NULL,
    ?EntityInterface $taxonomy_term = NULL,
    ?EntityInterface $user = NULL,
    ?EntityInterface $media = NULL,
    ?EntityInterface $block_content = NULL,
  ) {
    $this->entity = $node ?: $comment ?: $taxonomy_term ?: $user ?: $media ?: $block_content;

    if (!$this->entity) {
      $form['message'] = [
        '#markup' => $this->t('No entity found.'),
      ];
      return $form;
    }

    $servers = $this->configFactory->get('entity_io_push.settings')->get('servers') ?? [];

    if (empty($servers)) {
      $url = Url::fromRoute('entity_io_push.settings');
      $link = Link::fromTextAndUrl($this->t('configure them here'), $url)->toString();

      $form['message'] = [
        '#markup' => $this->t('No servers are configured. Please @link first.', ['@link' => $link]),
      ];

      return $form;
    }

    $options = [];
    foreach ($servers as $key => $server) {
      $options[$key] = $server['name'] . ' (' . $server['url'] . ')';
    }

    $form['server'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Server'),
      '#options' => $options,
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Push'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'deploy-result',
      ],
    ];

    $config = $this->configFactory->get('entity_io.user_field_settings');

    // Get base fields.
    $entity_type = $this->entityTypeManager->getDefinition('user');
    $base_fields = User::baseFieldDefinitions($entity_type);

    $bundle = 'user';

    $form['base_fields'] = [
      '#type' => 'details',
      '#title' => 'Base Fields',
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      $default = $config->get($bundle)['base_fields'][$base_field_name] ?? '';
      $form['base_fields'][$base_field_name] = [
        '#type' => 'checkbox',
        '#title' => $base_field_name,
        '#default_value' => $default,
      ];
    }

    $fields = $this->fieldConfigStorage->loadByProperties([
      'entity_type' => 'user',
      'bundle' => 'user',
    ]);

    if ($fields) {
      $form['fields'] = [
        '#type' => 'details',
        '#title' => 'Fields',
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      foreach ($fields as $field) {
        $default = $config->get($bundle)['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->getName(),
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => 'Check All',
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => 'Clear All',
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => 'Toggle',
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    // Depth select.
    $depth_options = [];
    for ($i = -1; $i <= 100; $i++) {
      $depth_options[(string) $i] = (string) $i;
    }

    $form['depth'] = [
      '#type' => 'select',
      '#title' => 'Entity reference depth',
      '#options' => $depth_options,
      '#default_value' => -1,
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'deploy-result'],
    ];

    return $form;
  }

  /**
   * Ajax submit handler.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): array {
    return $form['result'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $server_key = $form_state->getValue('server');

    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $max_depth = (int) $form_state->getValue('depth');
    $user = $this->getRequest()->attributes->get('user');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));
    $response = NULL;

    if ($user instanceof User) {
      $data = $this->entityIoExport::export($user, 0, $max_depth, $selected_fields, $current_langcode);
      $json = json_encode([$data]);
      $response = $this->pushService->push($json, $server_key);
    }

    $message = $response && $response['success']
      ? $this->t('Successfully deployed entity. Server response: @response', ['@response' => $response['body'] ?? 'No response'])
      : $this->t('Deployment failed: @message', ['@message' => $response['message'] ?? 'Unknown error']);

    $this->messenger->addMessage($message);
  }

}
