<?php

namespace Drupal\entity_io\Helper;

use Drupal\Core\Entity\EntityInterface;
use Drupal\block_content\Entity\BlockContent;
use Drupal\comment\Entity\Comment;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\taxonomy\Entity\Term;
use Drupal\user\Entity\User;

/**
 * Helper class to create entities from a JSON-like array structure.
 */
class EntityCreate {

  /**
   * Creates an entity from the provided JSON structure.
   *
   * @param string $type
   *   The entity type (e.g., 'node', 'taxonomy_term', 'user').
   * @param string $bundle
   *   The bundle or subtype of the entity (e.g., content type, vocabulary).
   * @param string $langcode
   *   The language code for the entity (e.g., 'en', 'pt-br').
   * @param array $json
   *   The structured array containing entity data.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The created entity object, or NULL if creation fails or type is invalid.
   */
  public static function create($type, $bundle, $langcode, array $json): ?EntityInterface {
    if (!$type || !$bundle || !$langcode) {
      return NULL;
    }

    // Determine which entity creation method to call based on entity type.
    switch ($type) {
      case 'block_content':
        return self::createBlockContent($json, $bundle, $langcode);

      case 'node':
        return self::createNode($json, $bundle, $langcode);

      case 'taxonomy_term':
        return self::createTerm($json, $bundle, $langcode);

      case 'paragraph':
        return self::createParagraph($bundle, $langcode);

      case 'user':
        return self::createUser($json, $langcode);

      case 'file':
        return self::createFile($json);

      case 'comment':
        return self::createComment($json, $bundle, $langcode);

      case 'media':
        // Handle different media bundles separately.
        if ($bundle == 'image') {
          return self::createMediaImage($json);
        }
        if ($bundle == 'audio') {
          return self::createMediaAudio($json);
        }
        if ($bundle == 'document') {
          return self::createMediaDocument($json);
        }
        if ($bundle == 'remote_video') {
          return self::createMediaRemoteVideo($json);
        }
        if ($bundle == 'video') {
          return self::createMediaVideo($json);
        }

      default:
        return NULL;
    }
  }

  /**
   * Creates a Comment entity.
   *
   * @param array $json
   *   The JSON-like structure with comment data.
   * @param string $bundle
   *   The comment type bundle.
   * @param string $langcode
   *   The language code.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The created Comment entity.
   */
  protected static function createComment(array $json, string $bundle, string $langcode): EntityInterface {

    if (isset($json['entity_id'][0]['target_id'])) {
      $entity_id = $json['entity_id'][0]['target_id'];
    }
    elseif (isset($json['entity_id'][0]['target__id'])) {
      $entity_id = $json['entity_id'][0]['target__id'];
    }

    $entity = Comment::create([
      'entity_type' => $json['entity_type'][0]['value'],
      'entity_id'   => $entity_id,
      'field_name'  => 'comment',
      'comment_type' => 'comment',
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a BlockContent entity.
   */
  protected static function createBlockContent(array $json, string $bundle, string $langcode): EntityInterface {
    $entity = BlockContent::create([
      'info' => $json['info'][0]['value'] ?? 'Block',
      'type' => $json['type'][0]['id'][0]['value'] ?? $bundle,
      'langcode' => $langcode,
      'reusable' => $json['reusable'][0]['value'] ?? TRUE,
      'body' => $json['body'][0]['value'] ?? '',
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a Node entity.
   */
  protected static function createNode(array $json, string $bundle, string $langcode): EntityInterface {
    $entity = Node::create([
      'type' => $bundle,
      'title' => $json['title'][0]['value'] ?? 'Untitled',
      'langcode' => $langcode,
      'status' => 1,
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a Taxonomy Term entity.
   */
  protected static function createTerm(array $json, string $bundle, string $langcode): EntityInterface {
    $entity = Term::create([
      'vid' => $bundle,
      'name' => $json['name'][0]['value'] ?? 'Unnamed term',
      'langcode' => $langcode,
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a Paragraph entity.
   */
  protected static function createParagraph(string $bundle, string $langcode): EntityInterface {
    $entity = Paragraph::create([
      'type' => $bundle,
      'langcode' => $langcode,
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a User entity.
   *
   * If a password hash is provided in the JSON, it updates the stored password
   * directly in the database.
   */
  protected static function createUser(array $json, string $langcode): EntityInterface {
    $entity = User::create([
      'name' => $json['name'][0]['value'] ?? 'user_' . time(),
      'mail' => $json['mail'][0]['value'] ?? 'user_' . time() . '@example.com',
      'pass' => $json['pass'][0]['value'] ?? 'defaultpass',
      'roles' => [$json['roles'][0]['target_id'] ?? 'authenticated'],
      'langcode' => $langcode,
      'status' => $json['status'][0]['value'] ?? 1,
    ]);

    // Directly update the password hash if provided.
    if (!empty($json['pass'][0]['value'])) {
      \Drupal::database()->update('users_field_data')
        ->fields(['pass' => $json['pass'][0]['value']])
        ->condition('uid', $entity->id())
        ->execute();
    }

    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates a File entity.
   */
  protected static function createFile(array $json): EntityInterface {
    $entity = File::create([
      'uri' => $json['uri'][0]['value'],
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);
    return $entity;
  }

  /**
   * Creates an Image Media entity.
   */
  protected static function createMediaImage(array $json): EntityInterface {
    $entity = Media::create([
      'bundle' => 'image',
      'name' => $json['name'][0]['value'],
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);

    return $entity;
  }

  /**
   * Creates an Audio Media entity.
   */
  protected static function createMediaAudio(array $json): EntityInterface {
    $entity = Media::create([
      'bundle' => 'audio',
      'name' => $json['name'][0]['value'],
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);

    return $entity;
  }

  /**
   * Creates a Document Media entity.
   */
  protected static function createMediaDocument(array $json): EntityInterface {
    $entity = Media::create([
      'bundle' => 'document',
      'name' => $json['name'][0]['value'],
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);

    return $entity;
  }

  /**
   * Creates a Remote Video Media entity.
   */
  protected static function createMediaRemoteVideo(array $json): EntityInterface {
    $entity = Media::create([
      'bundle' => 'remote_video',
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);

    return $entity;
  }

  /**
   * Creates a Video Media entity.
   */
  protected static function createMediaVideo(array $json): EntityInterface {
    $entity = Media::create([
      'bundle' => 'video',
    ]);
    \Drupal::state()->set('entity_io.skip_webhooks', TRUE);
    $entity->save();
    \Drupal::state()->delete('entity_io.skip_webhooks');
    $entity = EntityLoader::reloadEntity($entity);

    return $entity;
  }

}
