<?php

namespace Drupal\entity_io\Helper;

use Drupal\block_content\Entity\BlockContentType;
use Drupal\media\Entity\MediaType;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Helper class for parsing and processing entity data from JSON structures.
 *
 * Provides utility methods to:
 * - Extract metadata such as entity type, bundle, ID, and UUID.
 * - Handle default language adjustments to prevent invalid entity creation.
 */
class JsonParseEntityData {

  /**
   * Parses the JSON and extracts core entity metadata.
   *
   * Detects the entity type (e.g., node, taxonomy_term, media) from the JSON,
   * validates that the bundle/type exists in the system, and returns a metadata
   * array including entity type, bundle, ID, UUID, and optional title.
   *
   * @param array $json
   *   The structured array containing entity data.
   *
   * @return array
   *   An associative array with the following keys:
   *   - type: The entity type (e.g., 'node', 'taxonomy_term', 'media').
   *   - bundle: The entity bundle (e.g., content type, vocabulary,
   *     paragraph type).
   *   - id: The internal numeric ID of the entity.
   *   - uuid: The UUID string of the entity.
   *   - title: The entity title, if available (defaults to an empty string).
   *
   * @throws \RuntimeException
   *   Thrown if the entity type is unsupported or the specified
   *   bundle does not exist.
   */

  /**
   * Exported entities for reference during import.
   *
   * @var array
   */
  protected static $exportedEntities = [];

  /**
   * Parses the JSON and extracts core entity metadata.
   *
   * @var array
   */
  public static function parseEntityInfo(array $json): array {
    if (isset($json['exportedEntities'])) {
      self::$exportedEntities = $json['exportedEntities'];
    }

    if (isset($json['__entity_type__']) && $json['__entity_type__'] === 'node') {
      // Handle node entities.
      $type = 'node';
      $bundle = $json['type'][0]['id'][0]['value'];
      $id = $json['nid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];

      // Validate the node content type exists.
      if (!NodeType::load($bundle)) {
        throw new \RuntimeException("Content type '$bundle' doesn't exist.");
      }
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'taxonomy_term') {
      // Handle taxonomy term entities.
      $type = 'taxonomy_term';
      $bundle = $json['vid'][0]['vid'][0]['value'];
      $id = $json['tid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];

      // Validate the taxonomy vocabulary exists.
      if (!Vocabulary::load($bundle)) {
        throw new \RuntimeException("Taxonomy vocabulary '$bundle' doesn't exist.");
      }
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'media') {
      // Handle media entities.
      $type = 'media';
      $bundle = $json['bundle'][0]['id'][0]['value'];
      $id = $json['mid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];

      // Validate the media type exists.
      if (!MediaType::load($bundle)) {
        throw new \RuntimeException("Media type '$bundle' doesn't exist.");
      }
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'user') {
      // Handle user entities.
      $type = 'user';
      $bundle = 'user';
      $id = $json['uid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'comment') {
      // dd($json['uuid'][0]['value'], $json['cid'][0]['value']);
      // Handle comment entities.
      if (!isset($json['cid'][0]['value'])) {
        throw new \RuntimeException('Comment entity JSON is missing required fields: ' . json_encode($json));
      }
      $type = 'comment';
      $bundle = 'comment';
      $id = $json['cid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'file') {
      // Handle file entities.
      $type = 'file';
      $bundle = 'file';
      $id = $json['fid'][0]['value'];
      $uuid = $json['uuid'][0]['value'];
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'block_content') {
      // Handle custom block content entities.
      $type = 'block_content';
      $bundle = $json['type'][0]['id'][0]['value'];
      $id = $json['id'][0]['value'];
      $uuid = $json['uuid'][0]['value'];

      // Validate the block content type exists.
      if (!BlockContentType::load($bundle)) {
        throw new \RuntimeException("Block type '$bundle' doesn't exist.");
      }
    }
    elseif (isset($json['__entity_type__']) && $json['__entity_type__'] === 'paragraph') {
      // Handle paragraph entities.
      $type = 'paragraph';
      $bundle = $json['type'][0]['id'][0]['value'];
      $id = $json['id'][0]['value'];
      $uuid = $json['uuid'][0]['value'];

      // Validate the paragraph type exists.
      if (!ParagraphsType::load($bundle)) {
        throw new \RuntimeException("Paragraph type '$bundle' doesn't exist.");
      }
    }
    else {
      // Handle unsupported or unknown entity types.
      throw new \RuntimeException('Unrecognized or unsupported entity type in JSON.' . json_encode($json));
    }

    $available_languages = [];
    if (self::$exportedEntities && isset(self::$exportedEntities[$uuid])) {
      $available_languages = array_keys(self::$exportedEntities[$uuid]);
    }

    // Return the parsed entity metadata.
    return [
      'type' => $type,
      'bundle' => $bundle,
      'id' => $id,
      'uuid' => $uuid,
      'available_languages' => $available_languages,
      'title' => $json['title'][0]['value'] ?? '',
    ];
  }

  /**
   * Ensures entity creation uses the default language when needed.
   *
   * If the JSON indicates a language that is not the default language, this
   * method adjusts the JSON so that the default language version of the entity
   * is processed first, while preserving translations.
   *
   * @param array $json
   *   The structured JSON array containing entity data, including translations.
   * @param string|null $default_langcode
   *   The default language code to enforce. If NULL, the site's default.
   *
   * @return array
   *   The modified JSON data ensuring default language alignment, or the
   *   original JSON if no adjustments are required.
   */
  public static function preventEntityCreationLanguage($json, $default_langcode = NULL) {
    if (!$default_langcode) {
      $default_langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();
    }

    $exportedEntities = [];
    if (isset($json['exportedEntities'])) {
      $exportedEntities = $json['exportedEntities'];
    }

    // Check if translations exist and the entity language is not the default.
    if (isset($json['translations']) && isset($json['uuid'][0]['value']) && isset($json['langcode'][0]['value']) && $json['langcode'][0]['value'] != $default_langcode) {
      $new_json = [];
      // Start with the default language version of the entity.
      $new_json = $exportedEntities[$json['uuid'][0]['value']][$default_langcode];
      $new_json['exportedEntities'] = $exportedEntities;

      // Remove the default language from translations to avoid duplication.
      if (isset($exportedEntities[$json['uuid'][0]['value']][$default_langcode])) {
        unset($exportedEntities[$json['uuid'][0]['value']][$default_langcode]);
      }

      $new_json['translations'] = array_keys($exportedEntities[$json['uuid'][0]['value']]);

      return $new_json;
    }

    // Return the original JSON if no language adjustments are required.
    return $json;
  }

}
