<?php

namespace Drupal\entity_io_purge\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\entity_io\Helper\ExportDirectory;

/**
 * Form for selecting entity types to purge.
 */
class PurgeSelectionForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_purge_selection_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    // Load config and get available entity types from export directory.
    ExportDirectory::loadConfig();
    $files = ExportDirectory::getFiles('*');
    $entity_types = [];

    foreach ($files as $file_path) {
      // Extract entity type from file path.
      $publicPath = ExportDirectory::$publicPath;
      $relative_path = str_replace($publicPath . '/', '', $file_path);
      $parts = explode('/', $relative_path);
      if (!empty($parts[0])) {
        $entity_types[$parts[0]] = $parts[0];
      }
    }

    $form['fieldset_info'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Purge Exported Files'),
    ];

    if ($entity_types) {
      $form['fieldset_info']['description'] = [
        '#markup' => '<p>' . $this->t('Select which entity types you want to purge exported files for. This will permanently delete the JSON export files.') . '</p>',
      ];
    }

    if (empty($entity_types)) {
      $form['fieldset_info']['no_files'] = [
        '#markup' => '<p>' . $this->t('No exported files found.') . '</p>',
      ];
      return $form;
    }

    $form['fieldset_info']['directory_size'] = [
      '#markup' => '<p><strong>' . $this->t('Export directory size: @size MB', [
        '@size' => number_format((ExportDirectory::getDirectorySize() ?? 0) / 1024 / 1024, 2),
      ]) . '</strong></p>',
    ];

    $form['fieldset_info']['entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Entity types to purge'),
      '#options' => $entity_types,
      '#description' => $this->t('Select the entity types whose exported files you want to delete.'),
    ];

    $form['fieldset_info']['purge_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Purge all exported files'),
      '#description' => $this->t('Check this to delete all exported files regardless of entity type.'),
    ];

    $form['fieldset_info']['actions'] = [
      '#type' => 'actions',
    ];

    $form['fieldset_info']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Continue to confirmation'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $purge_all = $form_state->getValue('purge_all');

    if ($purge_all) {
      $entity_types = ['*'];
    }
    else {
      $entity_types = array_filter($form_state->getValue('entity_types'));
    }

    if (empty($entity_types)) {
      $this->messenger()->addError($this->t('Please select at least one entity type or check "Purge all".'));
      return;
    }

    $url = Url::fromRoute('entity_io_purge.confirm', [], [
      'query' => ['entity_types' => $entity_types],
    ]);

    $form_state->setRedirectUrl($url);
  }

}
