<?php

namespace Drupal\entity_io_webhooks\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service to upload JSON files via FTP.
 */
class FtpUploader {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a FtpUploader object.
   */
  public function __construct(ConfigFactoryInterface $configFactory, LoggerInterface $logger) {
    $this->configFactory = $configFactory;
    $this->logger = $logger;
  }

  /**
   * Envia um JSON para o FTP.
   */
  public function trigger($entity, string $filename, array $data): bool {
    $entity_type = $entity->getEntityTypeId();
    $bundle = method_exists($entity, 'bundle') ? $entity->bundle() : $entity_type;

    $config = $this->configFactory->get('entity_io_webhooks.settings');
    $bundle_cfg = $config->get("entities.$entity_type.$bundle") ?: [];

    // Verifica se a extensão FTP está habilitada.
    if (!function_exists('ftp_connect')) {
      $this->logger->error('A extensão FTP do PHP não está habilitada.');
      return FALSE;
    }

    $config = $this->configFactory->get('entity_io.settings');

    $host = $bundle_cfg['ftp']['ftp_host'];
    $user = $bundle_cfg['ftp']['ftp_user'];
    $pass = $bundle_cfg['ftp']['ftp_pass'];
    $path = rtrim($bundle_cfg['ftp']['ftp_path'] ?? '', '/');

    // Validação do host.
    if (empty($host) || !is_string($host)) {
      $this->logger->error('O host do FTP não está configurado corretamente.');
      return FALSE;
    }

    $ftp = ftp_connect($host, 21);
    if (!$ftp) {
      $this->logger->error('Não foi possível conectar ao FTP: @host', ['@host' => $host]);
      return FALSE;
    }

    if (!ftp_login($ftp, $user, $pass)) {
      $this->logger->error('Falha no login FTP para usuário @user', ['@user' => $user]);
      ftp_close($ftp);
      return FALSE;
    }

    ftp_pasv($ftp, TRUE);

    $remoteFile = $path ? "$path/" . basename($filename) : basename($filename);

    // Cria diretório remoto se não existir.
    if ($path) {
      $dirs = explode('/', $path);
      $currentDir = '';
      foreach ($dirs as $dir) {
        $currentDir .= ($currentDir ? '/' : '') . $dir;
        if (!@ftp_chdir($ftp, $currentDir)) {
          if (!@ftp_mkdir($ftp, $currentDir)) {
            $this->logger->error('Não foi possível criar o diretório remoto: @dir', ['@dir' => $currentDir]);
            ftp_close($ftp);
            return FALSE;
          }
        }
      }
      // Volta para o diretório inicial antes do upload.
      ftp_chdir($ftp, '/');
    }

    // Verifica se o arquivo existe antes de enviar.
    if (!file_exists($filename)) {
      $this->logger->error('Arquivo local não encontrado para upload: @file', ['@file' => $filename]);
      ftp_close($ftp);
      return FALSE;
    }

    // Use FTP_BINARY para arquivos JSON.
    $success = ftp_put($ftp, $remoteFile, $filename, FTP_BINARY);

    ftp_close($ftp);

    if ($success) {
      $this->logger->info('Arquivo JSON enviado com sucesso: @file', ['@file' => $remoteFile]);
    }
    else {
      $this->logger->error('Erro ao enviar arquivo JSON: @file', ['@file' => $remoteFile]);
    }

    return $success;
  }

}
