<?php

namespace Drupal\entity_io_webhooks\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Manager service to handle webhook triggering.
 */
class WebhooksManager {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a WebhooksManager object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger, ClientInterface $http_client, TimeInterface $time) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->httpClient = $http_client;
    $this->time = $time;
  }

  /**
   * Trigger webhooks for the given entity and event.
   */
  public function trigger($entity, $json, $event) {
    $entity_type = $entity->getEntityTypeId();
    $bundle = method_exists($entity, 'bundle') ? $entity->bundle() : $entity_type;

    $config = $this->configFactory->get('entity_io_webhooks.settings');
    $bundle_cfg = $config->get("entities.$entity_type.$bundle") ?: [];

    $events = $bundle_cfg['events'] ?? [];
    if (!in_array($event, $events)) {
      return;
    }

    $urls = isset($bundle_cfg['webhooks']['urls']) ? (array) $bundle_cfg['webhooks']['urls'] : [];
    if (empty($urls)) {
      $this->logger->warning('No webhook URL configured for @et:@bundle', ['@et' => $entity_type, '@bundle' => $bundle]);
      return;
    }

    // Prepare payload.
    $payload = [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
      'id' => $entity->id(),
      'event' => $event,
      'timestamp' => $this->time->getRequestTime(),
      'data' => $json,
    ];

    $headers = $config->get('default_headers') ?? '';

    $headers .= "\n" . $bundle_cfg['webhooks']['headers'];
    $headers_array = [];
    if (!empty($headers)) {
      // Parse lines "Key: Value".
      $lines = preg_split('/\r?\n/', $headers);
      foreach ($lines as $line) {
        if (strpos($line, ':') !== FALSE) {
          [$k, $v] = array_map('trim', explode(':', $line, 2));
          $headers_array[$k] = $v;
        }
      }
    }

    foreach ($urls as $url) {
      $url = trim($url);
      if (empty($url)) {
        continue;
      }
      try {
        $this->httpClient->request('POST', $url, [
          'json' => $payload,
          'headers' => $headers_array + ['Content-Type' => 'application/json'],
          'timeout' => 10,
        ]);
        $this->logger->info('Webhook sent to @url for @et:@id (@event)', [
          '@url' => $url,
          '@et' => $entity_type,
          '@id' => $entity->id(),
          '@event' => $event,
        ]);
      }
      catch (RequestException $e) {
        $this->logger->error('Webhook to @url failed: @m', [
          '@url' => $url,
          '@m' => $e->getMessage(),
        ]);
      }
      catch (\Exception $e) {
        $this->logger->error('Webhook to @url failed: @m', [
          '@url' => $url,
          '@m' => $e->getMessage(),
        ]);
      }
    }
  }

}
