<?php

namespace Drupal\entity_io\Controller;

use Drupal\Core\Entity\EntityInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for JSON API endpoints.
 */
class JsonApiController {

  /**
   * POST /api/receive-json.
   */
  public function receiveJson(Request $request): JsonResponse {
    // Check Basic Auth.
    $auth = $request->headers->get('Authorization');
    if (!$this->isAuthorized($auth)) {
      return new JsonResponse(['message' => 'Unauthorized'], Response::HTTP_UNAUTHORIZED);
    }

    $data = json_decode($request->getContent(), TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse(['message' => 'Invalid JSON'], Response::HTTP_BAD_REQUEST);
    }

    // Process the JSON.
    // Add your logic here, e.g., create or update content.
    \Drupal::logger('entity_io')->notice('Received data: <pre>@data</pre>', ['@data' => print_r($data, TRUE)]);

    return new JsonResponse(['message' => 'Data received successfully'], Response::HTTP_OK);
  }

  /**
   * GET /api/export-json/{bundle}/{id}.
   */
  public function exportJson(string $type, int $id, $revision = NULL): JsonResponse {
    $supported_types = ['node', 'taxonomy_term', 'media', 'comment', 'block_content', 'user', 'paragraph'];

    if (!in_array($type, $supported_types)) {
      return new JsonResponse(['message' => 'Unsupported entity type'], Response::HTTP_BAD_REQUEST);
    }

    $storage = \Drupal::entityTypeManager()->getStorage($type);

    if (!$storage) {
      return new JsonResponse(['message' => 'Invalid entity type storage'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    // Load either the default or revision.
    $entity = NULL;
    if ($revision && $storage->getEntityType()->isRevisionable()) {
      $entity = $storage->loadRevision($revision);
    }
    else {
      $entity = $storage->load($id);
    }

    if (!$entity instanceof EntityInterface) {
      return new JsonResponse(['message' => 'Entity not found'], Response::HTTP_NOT_FOUND);
    }

    $data = [
      'id' => $entity->id(),
      'uuid' => $entity->uuid(),
      'entity_type' => $type,
      'bundle' => method_exists($entity, 'bundle') ? $entity->bundle() : NULL,
      'revision_id' => method_exists($entity, 'getRevisionId') ? $entity->getRevisionId() : NULL,
      'langcode' => $entity->language()->getId(),
      'fields' => [],
    ];

    foreach ($entity->getFields() as $field_name => $field) {
      if (strpos($field_name, 'field_') === 0 || in_array($field_name, ['title', 'name', 'subject'])) {
        $data['fields'][$field_name] = $field->getValue();
      }
    }

    return new JsonResponse($data, Response::HTTP_OK);
  }

  /**
   * Check Basic Auth credentials.
   */
  private function isAuthorized(?string $auth): bool {
    if (strpos($auth, 'Basic ') !== 0) {
      return FALSE;
    }

    $encoded = substr($auth, 6);
    $decoded = base64_decode($encoded);
    [$user, $pass] = explode(':', $decoded, 2) + [NULL, NULL];

    $config = \Drupal::config('entity_io.settings');
    return $user === $config->get('api_user') && $pass === $config->get('api_pass');
  }

}
