<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\entity_io\Service\EntityIoExport;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Form to select what type of content to export.
 */
class EntityExportSelectorForm extends FormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity_io.export service (concrete/instance).
   *
   * @var \Drupal\entity_io\Service\EntityIoExport
   */
  protected $entityIoExport;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * Constructs the form.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, EntityIoExport $entity_io_export, $exporter) {
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->entityIoExport = $entity_io_export;
    $this->exporter = $exporter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('entity_io.export'),
      $container->get('entity_io.exporter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_entity_export_selector_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Use values from form_state when present (supports AJAX flow).
    $entity_type = $form_state->getValue('entity_type');
    $bundle = $form_state->getValue('bundle');

    // If no bundle value in form_state, keep any incoming default.
    // (this supports both initial render and AJAX-updated state).
    $current_bundle = $bundle ?? NULL;
    $selected_entity_id = $form_state->getValue('entity_id') ?? NULL;

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => [
        '' => $this->t('- Select -'),
        'node' => $this->t('Node'),
        'taxonomy_term' => $this->t('Taxonomy Term'),
        'block_content' => $this->t('Block'),
        'media' => $this->t('Media'),
        'user' => $this->t('User'),
      ],
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateBundleSelect',
        'wrapper' => 'bundle-wrapper',
      ],
      '#default_value' => $entity_type,
    ];

    $form['bundle_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'bundle-wrapper'],
    ];

    if (!empty($entity_type)) {
      $form['bundle_wrapper']['bundle'] = [
        '#type' => 'select',
        '#title' => $this->t('Filter by bundle'),
        '#options' => $this->getBundles($entity_type),
        '#ajax' => [
          'callback' => '::updateEntitySelect',
          'wrapper' => 'entity-wrapper',
        ],
        // Use the bundle value from form_state if present so AJAX works.
        '#default_value' => $current_bundle,
      ];
    }

    $form['entity_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'entity-wrapper'],
    ];

    if (!empty($entity_type)) {
      // Determine bundle to use for loading entities: prefer
      // current value from form_state.
      $bundle_to_use = $current_bundle;
      // Build options; if bundle not provided, getEntityOptions() may
      // return all or none per implementation.
      $options = $this->getEntityOptions($entity_type, $bundle_to_use);

      // If a submitted entity_id exists but is not present in $options,
      // load it and add it so validation passes.
      if (!empty($selected_entity_id) && !isset($options[$selected_entity_id])) {
        try {
          $storage = $this->entityTypeManager->getStorage($entity_type);
          $loaded_entity = $storage->load($selected_entity_id);
          if ($loaded_entity) {
            $options[$loaded_entity->id()] = $loaded_entity->label() ?: ('ID ' . $loaded_entity->id());
            // Keep keys sorted so select stays predictable (optional).
            ksort($options);
          }
        }
        catch (\Exception $e) {
          // Ignore - leave options as-is; validation
          // will handle missing entity.
        }
      }

      $form['entity_wrapper']['entity_id'] = [
        '#type' => 'select',
        '#title' => $this->t('Select specific item'),
        '#options' => $options,
        '#default_value' => $selected_entity_id,
      ];
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export'),
    ];

    return $form;
  }

  /**
   * Update the bundle select element.
   */
  public function updateBundleSelect(array &$form, FormStateInterface $form_state) {
    return $form['bundle_wrapper'];
  }

  /**
   * Update the entity select element.
   */
  public function updateEntitySelect(array &$form, FormStateInterface $form_state) {
    return $form['entity_wrapper'];
  }

  /**
   * Get entity options based on entity type and bundle.
   */
  protected function getEntityOptions($entity_type, $bundle) {
    $options = [];
    $storage = $this->entityTypeManager->getStorage($entity_type);

    try {
      switch ($entity_type) {
        case 'node':
          $entities = !empty($bundle)
            ? $storage->loadByProperties(['type' => $bundle])
            : $storage->loadMultiple();
          break;

        case 'taxonomy_term':
          $entities = !empty($bundle)
            ? $storage->loadByProperties(['vid' => $bundle])
            : $storage->loadMultiple();
          break;

        case 'block_content':
          $entities = !empty($bundle)
            ? $storage->loadByProperties(['type' => $bundle])
            : $storage->loadMultiple();
          break;

        case 'media':
          $entities = !empty($bundle)
            ? $storage->loadByProperties(['bundle' => $bundle])
            : $storage->loadMultiple();
          break;

        case 'user':
          $entities = $storage->loadMultiple();
          break;

        default:
          $entities = [];
      }
    }
    catch (\Exception $e) {
      // In case of unexpected storage/query errors, return empty options.
      return $options;
    }

    foreach ($entities as $entity) {
      $options[$entity->id()] = $entity->label() ?: ('ID ' . $entity->id());
    }

    return $options;
  }

  /**
   * Get bundles for a given entity type.
   */
  protected function getBundles($entity_type) {
    $options = [];

    switch ($entity_type) {
      case 'node':
        foreach ($this->entityTypeManager->getStorage('node_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'taxonomy_term':
        foreach ($this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple() as $vocab) {
          $options[$vocab->id()] = $vocab->label();
        }
        break;

      case 'media':
        foreach ($this->entityTypeManager->getStorage('media_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'block_content':
        foreach ($this->entityTypeManager->getStorage('block_content_type')->loadMultiple() as $type) {
          $options[$type->id()] = $type->label();
        }
        break;

      case 'user':
        $options = ['user' => 'User'];
        break;
    }

    return $options;
  }

  /**
   * Submit the export form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');
    $entity_id = $form_state->getValue('entity_id');

    $storage = $this->entityTypeManager->getStorage($entity_type);
    $entity = $storage->load($entity_id);

    if (!$entity) {
      $this->messenger()->addError($this->t('Entity not found.'));
      return;
    }

    $selected_fields = $this->entityIoExport->getSelectedFields($entity);
    $max_depth = -1;
    $depth = 0;
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $exporter = $this->exporter;
    $exporter::toJson($entity, $depth, $max_depth, $selected_fields, $current_langcode);

    // Retrieve JSON and filename from exporter class
    // (exporter implementation exposes these statics).
    $json = $exporter::$json ?? json_encode([]);
    $fileName = $exporter::$fileName ?? 'export.json';

    $response = new Response($json);
    $disposition = $response->headers->makeDisposition('attachment', $fileName);
    $response->headers->set('Content-Type', 'application/json');
    $response->headers->set('Content-Disposition', $disposition);

    $response->send();
    die;
  }

}
