<?php

namespace Drupal\entity_io\Form\Export;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\block_content\BlockContentInterface;
use Drupal\block_content\Entity\BlockContent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Block JSON export form.
 */
class BlockContentJsonExportForm extends FormBase {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The class resolver service.
   *
   * @var \Drupal\Core\DependencyInjection\ClassResolverInterface
   */
  protected $classResolver;

  /**
   * The exporter service (entity_io.exporter).
   *
   * @var mixed
   */
  protected $exporter;

  /**
   * Constructs the form.
   */
  public function __construct(LanguageManagerInterface $language_manager, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, ClassResolverInterface $class_resolver, $exporter) {
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->classResolver = $class_resolver;
    $this->exporter = $exporter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('class_resolver'),
      $container->get('entity_io.exporter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $block_content = NULL) {
    $language = $this->languageManager->getCurrentLanguage();

    if (!$block_content instanceof BlockContentInterface) {
      return ['error' => ['#markup' => $this->t('Invalid block')]];
    }

    if ($block_content->hasTranslation($language->getId())) {
      $block_content = $block_content->getTranslation($language->getId());
    }

    $config = $this->configFactory->get('entity_io.block_content_field_settings');

    $entity_type = $this->entityTypeManager->getDefinition('block_content');
    $base_fields = BlockContent::baseFieldDefinitions($entity_type);

    $form[$block_content->bundle()]['base_fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Base Fields'),
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      $default = $config->get($block_content->bundle())['base_fields'][$base_field_name] ?? '';
      $form[$block_content->bundle()]['base_fields'][$base_field_name] = [
        '#type' => 'checkbox',
        '#title' => $base_field_name,
        '#default_value' => $default,
      ];
    }

    // Load custom fields for this block content bundle.
    $fields = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
      'entity_type' => 'block_content',
      'bundle' => $block_content->bundle(),
    ]);

    if ($fields) {
      $form['fields'] = [
        '#type' => 'details',
        '#title' => $this->t('Fields'),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      // Use field objects to populate checkboxes.
      foreach ($fields as $field) {
        $default = $config->get($block_content->bundle())['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->getName(),
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => $this->t('Check All'),
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => $this->t('Clear All'),
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => $this->t('Toggle'),
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $options = [];
    for ($i = -1; $i <= 100; $i++) {
      $options[(string) $i] = (string) $i;
    }

    $form['depth'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity reference depth'),
      '#options' => $options,
      '#default_value' => $config->get($block_content->bundle())['depth'] ?? -1,
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['export'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export as JSON'),
    ];

    $form['queue_export'] = [
      '#type' => 'button',
      '#value' => $this->t('Add to Export Queue'),
      '#ajax' => [
        'callback' => '::ajaxQueueExport',
        'wrapper' => 'queue-export-messages',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding to queue...'),
        ],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['queue-export-btn'],
      ],
    ];

    $form['queue_messages'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $max_depth = (int) $form_state->getValue('depth');
    $block_content = $this->getRequest()->attributes->get('block_content');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    if ($block_content instanceof BlockContentInterface) {
      $exporter = $this->exporter;
      $exporter::toJson($block_content, 0, $max_depth, $selected_fields, $current_langcode);
      $json = $exporter::$json ?? json_encode([]);
      $fileName = $exporter::$fileName ?? 'export.json';

      $response = new Response($json);
      $response->headers->set('Content-Type', 'application/json');
      $response->headers->set('Content-Disposition', 'attachment; filename="' . $fileName . '"');
      $response->send();
      die;
    }
  }

  /**
   * AJAX callback for queue export button.
   */
  public function ajaxQueueExport(array &$form, FormStateInterface $form_state) {
    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields_bundle = $form_state->getValue($this->getRequest()->attributes->get('block_content')->bundle())['base_fields'] ?? [];
    $base_fields = array_filter($base_fields_bundle);
    $max_depth = (int) $form_state->getValue('depth');
    $block_content = $this->getRequest()->attributes->get('block_content');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));

    $response = [
      '#type' => 'container',
      '#attributes' => ['id' => 'queue-export-messages'],
    ];

    if ($block_content instanceof BlockContentInterface) {
      try {
        $data = [
          'entity_type' => $block_content->getEntityTypeId(),
          'entity_id' => $block_content->id(),
          'depth' => 0,
          'max_depth' => $max_depth,
          'selected_fields' => $selected_fields,
          'langcode' => $current_langcode,
          'is_revision' => FALSE,
        ];

        $controller = $this->classResolver->getInstanceFromDefinition('Drupal\entity_io_queue\Controller\EntityIoQueueController');
        $result = $controller->addExportItem($data);

        if ($result) {
          $response['message'] = [
            '#markup' => '<div class="messages messages--status">' .
            $this->t('Block content @title has been added to the export queue.', [
              '@title' => $block_content->label(),
            ]) . '</div>',
          ];
        }
        else {
          $response['message'] = [
            '#markup' => '<div class="messages messages--error">' .
            $this->t('Failed to add block content to queue.') . '</div>',
          ];
        }
      }
      catch (\Exception $e) {
        $response['message'] = [
          '#markup' => '<div class="messages messages--error">' .
          $this->t('Failed to add block content to queue: @message', [
            '@message' => $e->getMessage(),
          ]) . '</div>',
        ];
      }
    }
    else {
      $response['message'] = [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('Invalid block content entity.') . '</div>',
      ];
    }

    return $response;
  }

}
