<?php

namespace Drupal\entity_io\Helper;

/**
 * Helper class to work with entity fields by type, bundle, and field name.
 *
 * Provides utility methods to retrieve field definitions and properties such
 * as cardinality, translatability, type, and requirements for a given entity.
 */
class EntityFields {

  /**
   * Retrieves field definitions for a given entity type and bundle.
   *
   * @param string $type
   *   The entity type (e.g., 'node', 'taxonomy_term', 'paragraph').
   * @param string $bundle
   *   The entity bundle (e.g., content type, vocabulary, or paragraph type).
   * @param string|null $field_name
   *   Optional. The machine name of a specific field to retrieve.
   *
   * @return mixed
   *   - A single field definition object if $field_name is provided and exists.
   *   - An array of all field definitions.
   *   - NULL if no fields are found.
   */
  public static function get($type, $bundle, $field_name = NULL) {
    if ($type == 'file') {
      // Special case for file entity type which does not have bundles.
      $bundle = NULL;
    }

    if ($type == 'user') {
      // User entity type does not have bundles.
      $bundle = NULL;
    }

    $fields = \Drupal::service('entity_type.manager')->getStorage('field_config')->loadByProperties([
      'entity_type' => $type,
      'bundle' => $bundle,
    ]);

    $new_fields = [];
    foreach ($fields as $field) {
      $new_fields[$field->getName()] = $field;
    }
    $fields = $new_fields;

    if ($fields && $field_name && isset($fields[$field_name])) {
      return $fields[$field_name];
    }
    elseif ($fields) {
      return $fields;
    }

    return [];
  }

  /**
   * Retrieves field definitions for a given entity type and bundle.
   *
   * @param string $type
   *   The entity type (e.g., 'node', 'taxonomy_term', 'paragraph').
   * @param string $bundle
   *   The entity bundle (e.g., content type, vocabulary, or paragraph type).
   * @param string|null $field_name
   *   Optional. The machine name of a specific field to retrieve.
   *
   * @return mixed
   *   - A single field definition object if $field_name is provided and exists.
   *   - An array of all field definitions.
   *   - NULL if no fields are found.
   */
  public static function getAll($type, $bundle, $field_name = NULL) {
    $fields = self::get($type, $bundle);
    $base_fields = self::getBaseFields($type, $bundle);

    $fields = array_merge($fields, $base_fields ?? []);

    if ($fields && $field_name && isset($fields[$field_name])) {
      return $fields[$field_name];
    }
    elseif ($fields) {
      return $fields;
    }

    return [];
  }

  /**
   * Retrieves base field definitions for a given entity type and bundle.
   */
  public static function getBaseFields($type, $bundle = NULL, $field_name = NULL) {
    $fields = \Drupal::service('entity_field.manager')->getBaseFieldDefinitions($type, $bundle);

    if ($fields && $field_name && isset($fields[$field_name])) {
      return $fields[$field_name];
    }
    elseif ($fields) {
      return $fields;
    }

    return NULL;
  }

  /**
   * Retrieves the cardinality (number of allowed values) of a field.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   * @param string $field_name
   *   The machine name of the field.
   *
   * @return int|false
   *   The field cardinality as an integer, or FALSE if the field is not found.
   */
  public static function cardinality($type, $bundle, $field_name) {
    $field = self::get($type, $bundle, $field_name);
    if ($field) {
      return $field->getFieldStorageDefinition()->getCardinality();
    }
    return FALSE;
  }

  /**
   * Checks if a field is translatable.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   * @param string $field_name
   *   The machine name of the field.
   *
   * @return bool
   *   TRUE if the field is translatable, FALSE otherwise.
   */
  public static function isTranslatable($type, $bundle, $field_name) {
    $field = self::get($type, $bundle, $field_name);
    if ($field) {
      return $field->isTranslatable();
    }
    return FALSE;
  }

  /**
   * Checks if a field is required.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   * @param string $field_name
   *   The machine name of the field.
   *
   * @return bool
   *   TRUE if the field is required, FALSE otherwise.
   */
  public static function isRequired($type, $bundle, $field_name) {
    $field = self::get($type, $bundle, $field_name);
    if ($field) {
      return $field->isRequired();
    }
    return FALSE;
  }

  /**
   * Retrieves the type of a field (e.g., 'string', 'entity_reference').
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   * @param string $field_name
   *   The machine name of the field.
   *
   * @return string|false
   *   The field type as a string, or FALSE if the field is not found.
   */
  public static function type($type, $bundle, $field_name) {
    $field = self::get($type, $bundle, $field_name);
    if ($field) {
      return $field->getType();
    }
    return FALSE;
  }

  /**
   * Retrieves the target entity type for an entity reference field.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   * @param string $field_name
   *   The machine name of the field.
   *
   * @return string|false
   *   The target entity type as a string (e.g., 'node', 'taxonomy_term'),
   *   or FALSE if the field is not found or not an entity reference.
   */
  public static function targetType($type, $bundle, $field_name) {
    $field = self::get($type, $bundle, $field_name);
    if ($field) {
      return $field->getSetting('target_type');
    }
    return FALSE;
  }

  /**
   * Retrieves all required fields for a given entity type and bundle.
   *
   * @param string $type
   *   The entity type.
   * @param string $bundle
   *   The entity bundle.
   *
   * @return array
   *   An associative array of field definitions where the key is the field name
   *   and the value is the field definition object.
   */
  public static function getRequired($type, $bundle) {
    $fields = self::get($type, $bundle);
    $required = [];

    foreach ($fields as $field_name => $field) {
      if ($field->isRequired()) {
        $required[$field_name] = $field;
      }
    }

    return $required;
  }

}
