<?php

namespace Drupal\entity_io_purge\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\State\StateInterface;
use Drupal\entity_io\Helper\ExportDirectory;
use Psr\Log\LoggerInterface;

/**
 * Service to purge export storage directory.
 */
class PurgeService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  public function __construct(ConfigFactoryInterface $configFactory, FileSystemInterface $fileSystem, LoggerInterface $logger, StateInterface $state) {
    $this->configFactory = $configFactory;
    $this->fileSystem = $fileSystem;
    $this->logger = $logger;
    $this->state = $state;
  }

  /**
   * Clean up based on purge settings and cron frequency.
   */
  public function purge(): bool {
    $config = $this->configFactory->get('entity_io_purge.settings');

    // Check if auto purge is enabled.
    if (!$config->get('auto_purge')) {
      return FALSE;
    }

    // Check if it's time to run based on frequency.
    if (!$this->shouldRunPurge()) {
      return FALSE;
    }

    // Run the purge.
    $result = ExportDirectory::clearFiles();

    if ($result['deleted_count'] > 0) {
      $this->logger->notice('Entity IO purge service cleaned @count files.', [
        '@count' => $result['deleted_count'],
      ]);

      // Update last run time.
      $this->state->set('entity_io_purge.last_run', time());
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Check if purge should run based on frequency settings.
   */
  protected function shouldRunPurge(): bool {
    $config = $this->configFactory->get('entity_io_purge.settings');
    $frequency = $config->get('purge_frequency') ?? 'weekly';
    $last_run = $this->state->get('entity_io_purge.last_run', 0);
    $current_time = time();

    // Convert frequency to seconds.
    $frequency_seconds = $this->getFrequencyInSeconds($frequency);

    // Check if enough time has passed.
    return ($current_time - $last_run) >= $frequency_seconds;
  }

  /**
   * Convert frequency string to seconds.
   */
  protected function getFrequencyInSeconds(string $frequency): int {
    switch ($frequency) {
      case 'daily':
        // 24 hours
        return 86400;

      case 'weekly':
        // 7 days
        return 604800;

      case 'monthly':
        // ~30.44 days (average month)
        return 2629746;

      default:
        // Default to weekly.
        return 604800;
    }
  }

  /**
   * Manual purge that bypasses frequency checks.
   */
  public function manualPurge(): bool {
    $result = ExportDirectory::clearFiles();

    if ($result['deleted_count'] > 0) {
      $this->logger->notice('Manual Entity IO purge cleaned @count files.', [
        '@count' => $result['deleted_count'],
      ]);
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Purge files by entity types.
   *
   * @param array $entity_types
   *   Array of entity types to purge, or ['*'] for all.
   *
   * @return array
   *   Results array with deletion information.
   */
  public function purgeByEntityTypes(array $entity_types): array {
    $results = [];

    if (in_array('*', $entity_types)) {
      // Purge all files.
      $result = ExportDirectory::clearFiles();
      $results['all'] = $result;
    }
    else {
      // Purge specific entity types.
      foreach ($entity_types as $entity_type) {
        $result = ExportDirectory::clearFiles($entity_type);
        $results[$entity_type] = $result;
      }
    }

    return $results;
  }

  /**
   * Legacy purge method for backward compatibility.
   */
  public function legacyPurge(): bool {
    return $this->manualPurge();
  }

}
