<?php

namespace Drupal\entity_io_push\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\block_content\Entity\BlockContent;
use Drupal\entity_io\Service\EntityIoExportInterface;
use Drupal\entity_io_push\Service\JsonPushService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to deploy a block content entity JSON to a selected server.
 */
class DeployEntityBlockForm extends FormBase {

  /**
   * The JsonPushService.
   *
   * @var \Drupal\entity_io_push\Service\JsonPushService
   */
  protected $pushService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The entity being deployed.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $entity;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type repository.
   *
   * @var \Drupal\Core\Entity\EntityTypeRepositoryInterface
   */
  protected $entityTypeRepository;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity_io.export service.
   *
   * @var \Drupal\entity_io\Service\EntityIoExportInterface
   */
  protected $entityIoExport;

  /**
   * Constructs a DeployEntityBlockForm object.
   *
   * @param \Drupal\entity_io_push\Service\JsonPushService $push_service
   *   The JsonPushService service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeRepositoryInterface $entity_type_repository
   *   The entity type repository.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\entity_io\Service\EntityIoExportInterface $entity_io_export
   *   The entity_io.export service.
   */
  public function __construct(
    JsonPushService $push_service,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    EntityTypeManagerInterface $entity_type_manager,
    EntityTypeRepositoryInterface $entity_type_repository,
    LanguageManagerInterface $language_manager,
    EntityIoExportInterface $entity_io_export,
  ) {
    $this->pushService = $push_service;
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityTypeRepository = $entity_type_repository;
    $this->languageManager = $language_manager;
    $this->entityIoExport = $entity_io_export;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_io_push.json_push_service'),
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.repository'),
      $container->get('language_manager'),
      $container->get('entity_io.export')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_io_push_deploy_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?EntityInterface $node = NULL, ?EntityInterface $comment = NULL, ?EntityInterface $taxonomy_term = NULL, ?EntityInterface $user = NULL, ?EntityInterface $media = NULL, ?EntityInterface $block_content = NULL) {
    $this->entity = $node ?: $comment ?: $taxonomy_term ?: $user ?: $media ?: $block_content;

    if (!$this->entity) {
      $form['message'] = [
        '#markup' => $this->t('No entity found.'),
      ];
      return $form;
    }

    $servers = $this->configFactory->get('entity_io_push.settings')->get('servers') ?? [];

    if (empty($servers)) {
      $url = Url::fromRoute('entity_io_push.settings');
      $link = Link::fromTextAndUrl($this->t('configure them here'), $url)->toString();

      $form['message'] = [
        '#markup' => $this->t('No servers are configured. Please @link first.', ['@link' => $link]),
      ];

      return $form;
    }

    $options = [];
    foreach ($servers as $key => $server) {
      $options[$key] = $server['name'] . ' (' . $server['url'] . ')';
    }

    $form['server'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Server'),
      '#options' => $options,
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Push'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'deploy-result',
      ],
    ];

    // Use injected configFactory instead of \Drupal::config().
    $config = $this->configFactory->get('entity_io.block_content_field_settings');

    // Use injected entityTypeManager instead of \Drupal::entityTypeManager().
    $entity_type = $this->entityTypeManager->getDefinition('block_content');
    $base_fields = BlockContent::baseFieldDefinitions($entity_type);

    $form[$this->entity->bundle()]['base_fields'] = [
      '#type' => 'details',
      '#title' => 'Base Fields',
      '#tree' => TRUE,
    ];

    foreach ($base_fields as $base_field_name => $fieldDefinition) {
      $default = $config->get($this->entity->bundle())['base_fields'][$base_field_name] ?? '';
      $form[$this->entity->bundle()]['base_fields'][$base_field_name] = [
        '#type' => 'checkbox',
        '#title' => $base_field_name,
        '#default_value' => $default,
      ];
    }

    // Use injected entityTypeManager instead of \Drupal::entityTypeManager().
    $fields = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
      'entity_type' => 'block_content',
      'bundle' => $this->entity->bundle(),
    ]);

    if ($fields) {
      $form['fields'] = [
        '#type' => 'details',
        '#title' => 'Fields',
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      foreach ($fields as $field) {
        $default = $config->get($this->entity->bundle())['fields'][$field->getName()] ?? FALSE;
        $form['fields'][$field->getName()] = [
          '#type' => 'checkbox',
          '#title' => $field->label() . ' (' . $field->getName() . ')',
          '#default_value' => $default,
        ];
      }
    }

    $form['js_controls'] = [
      '#type' => 'container',
      'check_all' => [
        '#type' => 'button',
        '#value' => 'Check All',
        '#attributes' => ['class' => ['entity-io-check-all']],
      ],
      'clear_all' => [
        '#type' => 'button',
        '#value' => 'Clear All',
        '#attributes' => ['class' => ['entity-io-clear-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => 'Toggle',
        '#attributes' => ['class' => ['entity-io-toggle']],
      ],
    ];

    $options = [];
    for ($i = -1; $i <= 100; $i++) {
      $options[(string) $i] = (string) $i;
    }

    $form['depth'] = [
      '#type' => 'select',
      '#title' => 'Entity reference depth',
      '#options' => $options,
      '#default_value' => -1,
    ];

    $form['#attached']['library'][] = 'entity_io/entity_io_export';

    $form['result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'deploy-result'],
    ];

    return $form;
  }

  /**
   * Ajax callback for the submit button.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): array {
    return $form['result'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $server_key = $form_state->getValue('server');

    $fields = array_filter($form_state->getValue('fields') ?? []);
    $base_fields = array_filter($form_state->getValue('base_fields') ?? []);
    $max_depth = (Int) $form_state->getValue('depth');
    $block_content = $this->getRequest()->attributes->get('block_content');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    $selected_fields = array_merge(array_keys($fields), array_keys($base_fields));
    if ($block_content instanceof BlockContent) {
      $data = $this->entityIoExport::export($block_content, 0, $max_depth, $selected_fields, $current_langcode);
      $json = json_encode([$data]);
      $response = $this->pushService->push($json, $server_key);
    }

    if ($response['success']) {
      $message = $this->t('Successfully deployed entity. Server response: @response', [
        '@response' => $response['body'] ?? 'No response',
      ]);
      $this->messenger->addMessage($message);
    }
    else {
      $message = $this->t('Deployment failed: @message', [
        '@message' => $response['message'],
      ]);
      $this->messenger->addError($message);
    }

  }

}
