<?php

namespace Drupal\entity_io_push\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Service to push JSON to remote servers.
 */
class JsonPushService {

  /**
   * The HTTP client to make requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Push JSON data to a configured server.
   */
  public function push($json, string $server_name): array {
    $servers = $this->configFactory->get('entity_io_push.settings')->get('servers') ?? [];

    if (!isset($servers[$server_name])) {
      $this->logger->error('Server @server not found.', ['@server' => $server_name]);
      return ['success' => FALSE, 'message' => 'Server not found.'];
    }

    $server = $servers[$server_name];

    $endpoint = Url::fromRoute('entity_io_push.receive_json')->toString();

    $headers = [
      'Content-Type' => 'application/json',
    ];
    if (isset($server['headers']) && is_string($server['headers'])) {
      // Parse headers from string format (e.g., JSON or key:value pairs).
      $lines = explode("\n", $server['headers']);
      foreach ($lines as $line) {
        $parts = explode(':', $line, 2);
        if (count($parts) == 2) {
          $headers[trim($parts[0])] = trim($parts[1]);
        }
      }
    }

    try {
      $response = $this->httpClient->post($server['url'] . $endpoint, [
        'auth' => $server['user'] && $server['password'] ? [$server['user'], $server['password']] : NULL,
        'headers' => $headers,
        'body' => $json,
        'timeout' => 15,
      ]);

      return [
        'success' => TRUE,
        'status_code' => $response->getStatusCode(),
        'body' => $response->getBody()->getContents(),
      ];
    }
    catch (RequestException $e) {
      $this->logger->error('Push failed to @url: @message', [
        '@url' => $server['url'],
        '@message' => $e->getMessage(),
      ]);
      return ['success' => FALSE, 'message' => $e->getMessage()];
    }
  }

}
