<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;

/**
 * Form to select and batch export entities.
 */
class EntityBatchIdsExportForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_entity_batch_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => [
        '' => $this->t('- Select -'),
        'node' => $this->t('Node'),
        'taxonomy_term' => $this->t('Taxonomy Term'),
        'block_content' => $this->t('Block'),
        'media' => $this->t('Media'),
        'user' => $this->t('User'),
      ],
      '#required' => TRUE,
      '#default_value' => $entity_type,
    ];

    $form['ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('IDs.'),
      '#description' => $this->t('Enter the ids separated by comma.'),
      '#required' => TRUE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export All'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');
    $ids = $form_state->getValue('ids');

    if ($ids) {
      $ids = explode(",", $ids);
      $ids = array_filter($ids);
    }

    $batch_builder = new BatchBuilder();
    $batch_builder->setTitle($this->t('Exporting entities'))
      ->setFinishCallback([static::class, 'batchFinished']);

    foreach ($ids as $id) {
      $batch_builder->addOperation([static::class, 'exportEntityOperation'], [$entity_type, $id]);
    }

    batch_set($batch_builder->toArray());
  }

  /**
   * Export a single entity.
   */
  public static function exportEntityOperation($entity_type, $entity_id, array &$context) {
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    if ($entity) {
      $selected_fields = \Drupal::service('entity_io.export')::getSelectedFields($entity);
      $max_depth = -1;
      $depth = 0;
      $current_langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

      $exporter = \Drupal::service('entity_io.exporter');
      $exporter::toJson($entity, $depth, $max_depth, $selected_fields, $current_langcode);
      $file_url = $exporter::$publicUrl;

      $context['results'][] = Markup::create('<a href="' . $file_url . '" target="_blank">' . $file_url . '</a>');
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function batchFinished($success, array $results, array $operations) {
    if ($success) {
      \Drupal::messenger()->addMessage(t('Export finished. Download files:'));
      foreach ($results as $url) {
        \Drupal::messenger()->addMessage($url);
      }
    }
    else {
      \Drupal::messenger()->addError(t('Export failed.'));
    }
  }

}
