<?php

namespace Drupal\entity_io\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to filter entity IO logs.
 */
class EntityExportLogFilterForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new EntityExportLogFilterForm.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_io_log_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#method'] = 'get';
    $form['#disable_inline_form_errors'] = TRUE;

    $filters = $this->getRequest()->query->all();

    $form['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Filter logs'),
      '#open' => TRUE,
      '#wrapper_attributes' => ['class' => ['container-inline']],
    ];

    $user = isset($filters['uid']) ? $this->entityTypeManager->getStorage('user')->load($filters['uid']) : NULL;
    $form['filters']['uid'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('User'),
      '#target_type' => 'user',
      '#default_value' => $user,
      '#size' => 30,
    ];

    $operations = $this->database->select('entity_io_log', 'l')->distinct()->fields('l', ['operation'])->execute()->fetchCol();
    $form['filters']['operation'] = [
      '#type' => 'select',
      '#title' => $this->t('Operation'),
      '#options' => ['' => $this->t('- Any -')] + array_combine($operations, $operations),
      '#default_value' => $filters['operation'] ?? '',
    ];

    $entity_types = $this->database->select('entity_io_log', 'l')->distinct()->fields('l', ['entity_type'])->execute()->fetchCol();
    $form['filters']['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => ['' => $this->t('- Any -')] + array_combine($entity_types, $entity_types),
      '#default_value' => $filters['entity_type'] ?? '',
    ];

    $statuses = $this->database->select('entity_io_log', 'l')->distinct()->fields('l', ['status'])->execute()->fetchCol();
    $form['filters']['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#options' => ['' => $this->t('- Any -')] + array_combine($statuses, $statuses),
      '#default_value' => $filters['status'] ?? '',
    ];

    $form['filters']['message'] = [
      '#type' => 'search',
      '#title' => $this->t('Message'),
      '#default_value' => $filters['message'] ?? '',
      '#size' => 30,
    ];

    $form['filters']['actions'] = [
      '#type' => 'actions',
    ];

    $form['filters']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
    ];

    if (!empty($filters)) {
      $form['filters']['actions']['reset'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        '#submit' => ['::resetForm'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form is submitted via GET, so we don't need to do anything here.
  }

  /**
   * Resets the filter form.
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect($this->getRouteMatch()->getRouteName());
  }

}