<?php

namespace Drupal\entity_io\Helper;

use Drupal\Core\Entity\EntityInterface;

/**
 * Helper class to load entities by type, bundle, and UUID.
 */
class EntityLoader {

  /**
   * Loads an entity based on type, bundle, and UUID.
   *
   * @param array $params
   *   Associative array containing:
   *   - entity_type: The entity type (e.g., 'node', 'file', etc.)
   *   - entity_bundle: The bundle (e.g., content type, media type, etc.)
   *   - entity_uuid: The UUID of the entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The loaded entity or NULL if not found.
   */
  public static function loadEntity(array $params): ?EntityInterface {
    if (
      empty($params['entity_type']) ||
      empty($params['entity_bundle']) ||
      empty($params['entity_uuid'])
    ) {
      return NULL;
    }

    $entity_type = $params['entity_type'];
    $entity_uuid = $params['entity_uuid'];

    $entity_storage = \Drupal::entityTypeManager()->getStorage($entity_type);

    $entities = $entity_storage->loadByProperties([
      'uuid' => $entity_uuid,
    ]);

    return !empty($entities) ? reset($entities) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function reloadEntity(EntityInterface $entity) {
    $entity_type = $entity->getEntityTypeId();

    switch ($entity_type) {
      case 'user':
      case 'taxonomy_term':
      case 'paragraph':
      case 'node':
      case 'media':
      case 'comment':
      case 'block_content':
        $entity = \Drupal::entityTypeManager()
          ->getStorage($entity->getEntityTypeId())
          ->loadUnchanged($entity->id());
        break;
    }

    return $entity;
  }

}
